#include <stdlib.h>
#include <math.h>
#include <GL/glut.h>
#include "terrain.h"

double user_theta  = 0.3;
double user_height = 1.0;
double user_d_theta  = 0.0;
double user_d_height = 0.0;

void computeLocation() {
    double x = 2 * cos(user_theta);     // my x-, y-, and z-coordinates
    double y = 2 * sin(user_theta);
    double z = user_height;
    double d = sqrt(x * x + y * y + z * z); // distance to origin

    glMatrixMode(GL_PROJECTION);        // Set projection parameters.
    glLoadIdentity();
    glFrustum(-d * 0.3, d * 0.3, -d * 0.3, d * 0.3, d - 1.3, d + 1.3);
    gluLookAt(x, y, z,  0, 0, 0,  0, 0, 1);
}

// Initializes information for drawing within OpenGL.
void init() {
    GLfloat sun_direction[] = { 0.0, 1.0, 1.0, 0.0 };
    GLfloat sun_intensity[] = { 0.6, 0.6, 0.6, 1.0 };
    GLfloat amb_intensity[] = { 0.4, 0.4, 0.4, 1.0 };

    glClearColor(0.5, 0.5, 1.0, 0.0);   // Set window color to sky blue.
    computeLocation();

    glEnable(GL_DEPTH_TEST);            // Draw only closest surfaces

    glEnable(GL_LIGHTING);              // Set up ambient light.
    glLightModelfv(GL_LIGHT_MODEL_AMBIENT, amb_intensity);

    glEnable(GL_LIGHT0);                // Set up sunlight.
    glLightfv(GL_LIGHT0, GL_POSITION, sun_direction);
    glLightfv(GL_LIGHT0, GL_DIFFUSE, sun_intensity);

    glEnable(GL_COLOR_MATERIAL);        // Configure glColor().
    glColorMaterial(GL_FRONT, GL_AMBIENT_AND_DIFFUSE);

    terrainInit();
}

// Draws the current image.
void draw() {
    glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT); // Clear window.
    glColor3f(1.0, 1.0, 1.0);
    glShadeModel(GL_SMOOTH);
    glPushMatrix();
    glTranslatef(-1.0, -1.0, 0.0);
    glScalef(2.0, 2.0, 1.0);
    terrainDraw();
    glPopMatrix();
    glutSwapBuffers();
}

// Arranges that the window will be redrawn roughly every 40 ms.
void idle() {
    static int lastTime = 0;                // time of last redraw
    int time = glutGet(GLUT_ELAPSED_TIME);  // current time

    if(lastTime == 0 || time >= lastTime + 40) {
        lastTime = time;
        if(user_d_theta != 0.0 || user_d_height != 0.0) {
            user_theta += user_d_theta;
            user_height += user_d_height;
            computeLocation();
            glutPostRedisplay();
        }
    }
}

// When window becomes visible, we want the window to
// continuously repaint itself.
void visible(int vis) {
    glutIdleFunc(vis == GLUT_VISIBLE ? idle : NULL);
}

// Called when a "special" key is pressed
void special(int k, int x, int y) {
    switch(k) {
    case GLUT_KEY_UP:    user_d_height += 0.05; break;
    case GLUT_KEY_DOWN:  user_d_height -= 0.05; break;
    case GLUT_KEY_LEFT:  user_d_theta  += 0.05; break;
    case GLUT_KEY_RIGHT: user_d_theta  -= 0.05; break;
    }
    if(fabs(user_d_height) < 0.02) user_d_height = 0.0;
    if(user_d_height > 1.0) user_d_height = 1.0;
    if(user_d_height < -1.0) user_d_height = -1.0;
    if(fabs(user_d_theta) < 0.02)  user_d_theta = 0.0;
    if(user_d_theta > 1.0) user_d_theta = 1.0;
    if(user_d_theta < -1.0) user_d_theta = -1.0;
}

int main(int argc, char **argv) {
    glutInit(&argc, argv);
    glutInitDisplayMode(GLUT_DEPTH | GLUT_DOUBLE | GLUT_RGB);
    glutInitWindowPosition(50, 100);    // Set up display window.
    glutInitWindowSize(300, 300);
    glutCreateWindow("Island");
    srandom(time(NULL));

    init();
    glutDisplayFunc(draw);
    glutVisibilityFunc(visible);
    glutSpecialFunc(special);
    glutMainLoop();
    return 0;
}
