import java.awt.Graphics;
import java.awt.Color;

public abstract class Model {
	public static final Model EMPTY = new Empty();

	private static class Empty extends Model {
		public void draw(Graphics3D g) { }
		public Model transform(Transform xform) { return this; }
	}

	private static class TransformedModel extends Model {
		private Model model;
		private Transform x;

		public TransformedModel(Model model, Transform xform) {
			this.model = model;
			this.x = xform;
		}

		public void draw(Graphics3D g) {
			Transform old = g.getModelTransform();
			g.setModelTransform(old.append(x));
			model.draw(g);
			g.setModelTransform(old);
		}

		public Model transform(Transform xform) {
			return new TransformedModel(model, xform.append(x));
		}
	}

	private static class ColoredModel extends Model {
		private Model model;
		private Color color;
		public ColoredModel(Model m, Color c) { model = m; color = c; }
		public void draw(Graphics3D g) {
			Color old = g.getColor();
			g.setColor(color);
			model.draw(g);
			g.setColor(old);
		}
	}

	public abstract void draw(Graphics3D g);

	public Model color(Color c) {
		return new ColoredModel(this, c);
	}

	public Model transform(Transform xform) {
		return new TransformedModel(this, xform);
	}

	public Model scale(double x, double y, double z) {
		return transform(Transform.scale(x, y, z));
	}
	public Model translate(double x, double y, double z) {
		return transform(Transform.translate(x, y, z));
	}
	public Model rotateX(double theta) {
		return transform(Transform.rotateX(theta));
	}
	public Model rotateY(double theta) {
		return transform(Transform.rotateY(theta));
	}
	public Model rotateZ(double theta) {
		return transform(Transform.rotateZ(theta));
	}
}
