import java.text.DecimalFormat;

public class Transform {
	public static final Transform IDENTITY = new Transform(
			new double[] { 1, 0, 0, 0,
				           0, 1, 0, 0,
						   0, 0, 1, 0,
						   0, 0, 0, 1 });
	private static DecimalFormat form = new DecimalFormat("#0.00");

	private static final int SIZE = 4;

	private double[] data;

	private Transform(double[] data) {
		this.data = data;
	}

	public String toString() {
		StringBuffer buf = new StringBuffer("^");
		for(int i = 0; i < data.length; i++) {
			if(i % SIZE == 0) {
				if(i > 0) buf.append("]\n ");
				buf.append("[");
			} else {
				buf.append(",");
			}
			buf.append(form.format(data[i]));
		}
		buf.append("]");
		return buf.toString();
	}

	public Transform append(Transform other) {
		double[] ret = new double[SIZE * SIZE];
		for(int r = 0; r < ret.length; r++) {
			double val = 0.0;
			for(int a = (r / SIZE) * SIZE, b = r % SIZE;
					b < SIZE * SIZE;
					a++, b += SIZE) {
				val += this.data[a] * other.data[b];
			}
			ret[r] = val;
		}
		return new Transform(ret);
	}

	public Transform prepend(Transform other) {
		return other.append(this);
	}

	public Point transform(Point p) {
		double x = p.getX();
		double y = p.getY();
		double z = p.getZ();
		double h = p.getHomogeneous();

		double a = data[ 0] * x + data[ 1] * y + data[ 2] * z
			+ data[ 3] * h;
		double b = data[ 4] * x + data[ 5] * y + data[ 6] * z
			+ data[ 7] * h;
		double c = data[ 8] * x + data[ 9] * y + data[10] * z
			+ data[11] * h;
		double s = data[12] * x + data[13] * y + data[14] * z
			+ data[15] * h;

		return Point.create(a, b, c, s);
	}

	public Vector transform(Vector v) {
		double x = v.getX();
		double y = v.getY();
		double z = v.getZ();

		double a = data[ 0] * x + data[ 1] * y + data[ 2] * z;
		double b = data[ 4] * x + data[ 5] * y + data[ 6] * z;
		double c = data[ 8] * x + data[ 9] * y + data[10] * z;

		return Vector.create(a, b, c);
	}

	public static Transform create(double[] mat) {
		return new Transform(mat);
	}

	public static Transform translate(double x, double y, double z) {
		return new Transform(new double[] {
			1, 0, 0, x,
			0, 1, 0, y,
			0, 0, 1, z,
			0, 0, 0, 1 });
	}

	public static Transform scale(double x, double y, double z) {
		return new Transform(new double[] {
			x, 0, 0, 0,
			0, y, 0, 0,
			0, 0, z, 0,
			0, 0, 0, 1 });
	}

	public static Transform rotateX(double theta) {
		double c = Math.cos(theta);
		double s = Math.sin(theta);
		return new Transform(new double[] {
			1, 0, 0, 0,
			0, c,-s, 0,
			0, s, c, 0,
			0, 0, 0, 1 });
	}

	public static Transform rotateY(double theta) {
		double c = Math.cos(theta);
		double s = Math.sin(theta);
		return new Transform(new double[] {
			 c, 0, s, 0,
			 0, 1, 0, 0,
			-s, 0, c, 0,
			 0, 0, 0, 1 });
	}

	public static Transform rotateZ(double theta) {
		double c = Math.cos(theta);
		double s = Math.sin(theta);
		return new Transform(new double[] {
			c,-s, 0, 0,
			s, c, 0, 0,
			0, 0, 1, 0,
			0, 0, 0, 1 });
	}

}
