package csbsju.cs160;

import java.awt.Dimension;
import java.awt.Image;
import java.awt.event.*;
import javax.swing.*;
import java.util.*;

/**
 * Provides a window with a blank canvas into which a
 * program might draw. For many features of the window,
 * you'll have to extend this class and override the
 * methods. You need to do this, for example, if you
 * want to add your own menu items.
 *
 * @author  Carl Burch
 * @version 28 August 2001
 */
public class DrawableFrame {	
	private class MyFrame extends JFrame
			implements WindowListener, Runnable {
		public MyFrame(String title) {
			super(title);
			addWindowListener(this);
			MyFrame.this.setBackground(java.awt.Color.white);
		}
		public void run() { DrawableFrame.this.run(); }
		public void windowActivated(WindowEvent e) { }
		public void windowClosed(WindowEvent e) {
			System.exit(0);
			// frame = null;
		}
		public void windowClosing(WindowEvent e) { this.dispose(); }
		public void windowDeactivated(WindowEvent e) { }
		public void windowDeiconified(WindowEvent e) { }
		public void windowIconified(WindowEvent e) { }
		public void windowOpened(WindowEvent e) { }
  	}

	private class FrameMenuItem extends JMenuItem implements ActionListener {
		public FrameMenuItem(String name) {
			super(name);
			addActionListener(this);
		}

		public void actionPerformed(ActionEvent e) {
			menuItemSelected(getText());
		}
	}

	private class DrawablePanel extends JPanel
			implements MouseListener, MouseMotionListener, KeyListener {
		Image buffer = null;
		Dimension lastSize = null;
		private int last_x;
		private int last_y;

		public DrawablePanel(int width, int height) {
			setPreferredSize(new Dimension(width, height));
			DrawablePanel.this.setBackground(java.awt.Color.white);
			addMouseListener(this);
			addMouseMotionListener(this);
			addKeyListener(this);
		}

		public void setBackground(java.awt.Color color) {
			super.setBackground(color);
			buffer = null;
		}

		public java.awt.Graphics getGraphicsContext() {
			Dimension sz = getSize();
			if(lastSize == null || sz.width != lastSize.width || sz.height != lastSize.height) {
				lastSize = new Dimension(sz);
				Image newbuf = createImage(sz.width, sz.height);
				java.awt.Graphics g = newbuf.getGraphics();
				g.setColor(java.awt.Color.white);
				g.fillRect(0, 0, sz.width, sz.height);
				g.setColor(java.awt.Color.black);
				if(buffer != null) {
					g.drawImage(buffer, 0, 0, null);
				}
				buffer = newbuf;
			}
			this.repaint();
			return buffer.getGraphics();
		}

		public void paintComponent(java.awt.Graphics g) {
			super.paintComponent(g);
			Dimension sz = getSize();
			if(buffer != null) g.drawImage(buffer, 0, 0, null);
		}

		public void mouseClicked(MouseEvent e) { }
		public void mouseEntered(MouseEvent e) { }
		public void mouseExited(MouseEvent e) { }
		public void mousePressed(MouseEvent e) {
			grabFocus();
			last_x = e.getX();
			last_y = e.getY();
			DrawableFrame.this.mouseDragging(last_x, last_y,
				last_x, last_y);
		}
		public void mouseReleased(MouseEvent e) {
			DrawableFrame.this.mouseDragged(last_x, last_y,
				e.getX(), e.getY());
		}
		public void mouseMoved(MouseEvent e) { }
		public void mouseDragged(MouseEvent e) {
			DrawableFrame.this.mouseDragging(last_x, last_y,
				e.getX(), e.getY());
		}

		public void keyPressed(KeyEvent e) {
			DrawableFrame.this.keyPressed(e.getKeyChar());
		}
		public void keyReleased(KeyEvent e) { }
		public void keyTyped(KeyEvent e) { }
	}

	private MyFrame frame = null;
	private JMenuBar menubar = new JMenuBar();
	private LinkedList menus = new LinkedList();
	private DrawablePanel panel = null;

	/** 
	 * Creates new DrawableFrame whose canvas is 200 pixels in
	 * height and width.
	 */
	public DrawableFrame() {
		this(200, 200);
	}

	/** 
	 * Creates new DrawableFrame whose canvas is <code>width</code>
	 * pixels wide and <code>height</code> pixels tall.
	 */
    public DrawableFrame(int width, int height) {
		frame = new MyFrame("Drawable Frame");
		frame.setJMenuBar(menubar);

		JMenu file_menu = new JMenu("File");
		menus.add(file_menu);
		menubar.add(file_menu);

		// JMenuItem item = new JMenuItem("Close");
		JMenuItem item = new JMenuItem("Quit");
		file_menu.add(item);
		item.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				DrawableFrame.this.dispose();
			}
		});

		panel = new DrawablePanel(width, height);
		frame.getContentPane().add(panel);
		frame.pack();
    }

	/**
	 * Returns the width in pixels of the drawing area.
	 */
	public int getWidth() {
		return panel.getWidth();
	}

	/**
	 * Returns the height in pixels of the drawing area.
	 */
	public int getHeight() {
		return panel.getHeight();
	}

	/**
	 * Changes the background color of the window. Note that
	 * this clears the canvas and all previous <code>Graphics</code>
	 * objects returned by <code>getGraphics()</code> are 
	 * rendered ineffective by this operation.
	 */
	public void setBackground(Color color) {
		frame.setBackground(color.getColor());
		panel.setBackground(color.getColor());
	}

	/**
	 * Returns a Graphics object that you can use for drawing in the
	 * drawing area.
	 */
	public Graphics getGraphics() {
		return new Graphics(panel.getGraphicsContext(), panel);
	}

	/**
	 * Returns a Pen object that you can use for drawing in the
	 * drawing area, initialized to begin at location
	 * (<code>x</code>,<code>y</code>) in the drawing area.
	 */
	public Pen getPen(double x, double y) {
		return new Pen(getGraphics(), x, y);
	}

	/**
	 * Handles the event when the mouse button is pressed at location
	 * (x0,y0) in the drawing area, is currently at 
	 * (x1,y1), and has not yet been released. This method will be
	 * called repeatedly; when the user releases the mouse, the
	 * <code>mouseDragged()</code> method is called. By default the
	 * <code>mouseDragging()</code> method does
	 * nothing, but you can define it in your class to do something useful.
	 */
	public void mouseDragging(int x0, int y0, int x1, int y1) { }

	/**
	 * Handles the event when the mouse button is pressed at location
	 * (x0,y0) in the drawing area and subsequently released at location
	 * (x1,y1). By default this does nothing, but
	 * you can define it in your class to do something useful.
	 */
	public void mouseDragged(int x0, int y0, int x1, int y1) { }

	/**
	 * Handles the event when a keyboard key is pressed.
	 * By default this does nothing, but
	 * you can define it in your class to do something useful.
	 */
	public void keyPressed(char c) { }

	/**
	 * Shows the window to the user. You should call this to start the
	 * graphics interface.
	 */
	public void show() {
		frame.pack();
		frame.show();
		new Thread(frame).start();
		panel.grabFocus();
	}

	/**
	 * Removes the window from the screen.
	 */
	public void dispose() {
		frame.dispose();
	}

	/**
	 * Sets the title appearing in the title bar.
	 */
	public void setTitle(String title) {
		frame.setTitle(title);
	}

	/**
	 * Adds a menu with the given title into the window's menu bar.
	 */
	public void addMenu(String title) {
		JMenu menu = new JMenu(title);
		menus.add(menu);
		menubar.add(menu);
	}

	/**
	 * Adds a menu item titled as in the second parameter into the
	 * menu whose title is as specified in the first parameter.
	 * When this menu item is selected, the
	 * <code>menuItemSelected()</code> method is called.
	 * You will want to override this method if you want the
	 * menu item to do something useful.
	 */
	public void addMenuItem(String menu, String title) {
		FrameMenuItem item = new FrameMenuItem(title);
		JMenu m = null;
		Iterator it = menus.iterator();
		while(m == null && it.hasNext()) {
			JMenu me = (JMenu) it.next();
			if(menu.equals(me.getText())) m = me;
		}
		if(m == null) throw new Error("Menu `" + menu + "' doesn't exist.");
		m.add(item);
	}

	/**
	 * Handles the event when the user selects an item of the given
	 * title from a menu. By default this does nothing, but you can
	 * define it in your class to do something useful.
	 */
	public void menuItemSelected(String title) { }

	/**
	 * Run the program after the window appears on the screen. By
	 * default, this does nothing; but you can define your own version
	 * if you want something to happen as soon as the window is
	 * displayed. This method can last as long as you want;
	 * this is especially useful when you want to do some
	 * animation.
	 */
	public synchronized void run() { }
}
