package csbsju.cs160;

import javax.swing.*;
import java.awt.*;
import java.awt.event.*;
import java.awt.datatransfer.*;
import java.util.*;

/**
 * Provides a set of static methods for printing and reading
 * from a separate window.
 *
 * @author  Carl Burch
 * @version 28 August 2001
 */
public class IO {
	private IO() { }

	private static final boolean DR_JAVA = false;
	private static MyFrame frame = null;
	private static InputBuffer buffer = new InputBuffer();

	private static class IOArea extends JPanel
			implements KeyListener, AdjustmentListener {
		protected static final char HUMAN = '\001';
		protected static final char COMPUTER = '\002';
		protected static final String PROMPT = "? " + HUMAN;
		protected static final int NUM_ROWS = 10;
		protected static final int NUM_COLS = 50;

		protected java.awt.Font computer_font = new Font("Monospaced", Font.PLAIN, 12);
		protected java.awt.Font user_font = new Font("Monospaced", Font.BOLD, 12);
		protected java.awt.Color computer_color = java.awt.Color.black;
		protected java.awt.Color user_color = java.awt.Color.blue;

		protected class IOCanvas extends JPanel
		implements MouseListener, FocusListener, ComponentListener {
			public int top_row = 0;

			public IOCanvas() {
				setBackground(java.awt.Color.white);
				addMouseListener(this);
				addFocusListener(this);
				addComponentListener(this);
				setSize(NUM_COLS, NUM_ROWS);
			}

			public void setSize(int cols, int rows) {
				java.awt.geom.Rectangle2D rect
					= computer_font.getStringBounds("x",
						new java.awt.font.FontRenderContext(null, false, false));
				setPreferredSize(new Dimension((int) (cols * rect.getWidth()),
					(int) (rows * rect.getHeight())));
			}
			public int getRows() {
				java.awt.geom.Rectangle2D rect
					= computer_font.getStringBounds("x",
						new java.awt.font.FontRenderContext(null, false, false));
				Dimension sz = getSize();
				return (int) (sz.height / rect.getHeight());
			}

			public void paintComponent(java.awt.Graphics g) {
				super.paintComponent(g);
				g.setFont(computer_font);
				FontMetrics fm = g.getFontMetrics();
				int line_width = 0;
				int y = fm.getLeading() + fm.getAscent();

				// display the committed lines
				for(int i = top_row; i < text.size(); i++) {
					String line = (String) text.get(i);
					line_width = paintLine(g, line, 0, y);
					y += fm.getHeight();
				}

				// display lines in the buffer
				y -= fm.getHeight();
				StringTokenizer buf_lines = new StringTokenizer(buffer, "\n", true);
				while(buf_lines.hasMoreTokens()) {
					String line = buf_lines.nextToken();
					if(line.equals("\n")) {
						y += fm.getHeight();
						line_width = 0;
					} else {
						line_width = paintLine(g, "" + HUMAN + line, line_width, y);
					}
				}

				// display the cursor
				if(hasFocus()) {
					g.drawLine(line_width, y, line_width, y - fm.getAscent());
				}
			}
			protected int paintLine(java.awt.Graphics g, String line, int x, int y) {
				char mode = COMPUTER;
				g.setFont(computer_font);
				g.setColor(computer_color);
				int pos = 0;
				while(pos < line.length()) {
					// set font for current mode
					g.setFont(mode == COMPUTER ? computer_font : user_font);
					g.setColor(mode == COMPUTER ? computer_color : user_color);
					FontMetrics fm = g.getFontMetrics();

					// compute end of fragment and update mode
					int next = line.length();
					int k = line.indexOf(COMPUTER, pos);
					if(k >= 0 && k < next) { next = k; mode = COMPUTER; }
					k = line.indexOf(HUMAN, pos);
					if(k >= 0 && k < next) { next = k; mode = HUMAN; }

					// draw fragment and step up to next position
					String frag = line.substring(pos, next);
					g.drawString(frag, x, y);
					x += fm.stringWidth(frag);
					pos = next + 1;
				}
				return x;
			}

			//
			// methods as MouseListener only
			//
			public void mouseClicked(MouseEvent e) { this.grabFocus(); }
			public void mouseEntered(MouseEvent e) { }
			public void mouseExited(MouseEvent e) { }
			public void mousePressed(MouseEvent e) { }
			public void mouseReleased(MouseEvent e) { }

			//
			// methods as FocusListener
			//
			public void focusLost(java.awt.event.FocusEvent p1) { repaint(); }
			public void focusGained(java.awt.event.FocusEvent p1) { repaint(); }

			//
			// methods as ComponentListener
			//
			public void componentHidden(ComponentEvent e) { }
			public void componentMoved(ComponentEvent e) { }
			public void componentResized(ComponentEvent e) {
				scrollbar.update();
			}
			public void componentShown(ComponentEvent e) { }
		}

		protected class IOScrollBar extends JScrollBar {
			public IOScrollBar() {
				setMinimum(0);
				setUnitIncrement(1);
				setBlockIncrement(IOArea.NUM_ROWS);
				setVisibleAmount(IOArea.NUM_ROWS);
				update();
			}
			public void update() {
				boolean at_end = getValue() == getMaximum();
				int old_value = getValue();
				int rows = canvas.getRows();
				int lines = text.size();
				lines += countOccurrences(buffer, '\n');
				if(lines < rows) lines = rows;
				setValue(0);
				setBlockIncrement(rows);
				setVisibleAmount(rows);
				setMaximum(lines);
				if(at_end || old_value > getMaximum()) {
					setValue(getMaximum());
				} else {
					setValue(old_value);
				}
			}
		}

		protected java.util.Vector text = new java.util.Vector();
		protected String buffer = "";
		protected IOCanvas canvas;
		protected IOScrollBar scrollbar;

		public IOArea() {
			super(new BorderLayout());
			// setBorder(javax.swing.border.LineBorder.createBlackLineBorder());
			setBorder(new javax.swing.border.EtchedBorder());
			add(canvas = new IOCanvas(), BorderLayout.CENTER);
			add(scrollbar = new IOScrollBar(), BorderLayout.EAST);

			text.add("");

			canvas.addKeyListener(this);
			scrollbar.addAdjustmentListener(this);
		}

		public void clear() {
			text = new java.util.Vector();
			buffer = "";
			scrollbar.update();
			canvas.repaint();
		}

		public void setSize(int cols, int rows) {
			canvas.setSize(cols, rows);
		}
		public void print(String what) {
			moveToBottom();
			addText(COMPUTER, what);
		}
		public synchronized String readLine() {
			moveToBottom();
			canvas.grabFocus();

			while(available() == 0) {
				try {
					wait();
				} catch(InterruptedException e) { }
			}

			int pos = buffer.indexOf('\n');
			String ret = buffer.substring(0, pos);
			buffer = buffer.substring(pos + 1);
			addText(HUMAN, ret + '\n');
			return ret;
		}
		public int available() {
			return buffer.indexOf('\n') < 0 ? 0 : buffer.length();
		}

		public void grabFocus() {
			canvas.grabFocus();
		}
		protected void moveToBottom() {
			scrollbar.setValue(scrollbar.getMaximum());
		}

		protected String getText() {
			StringBuffer ret = new StringBuffer();
			java.util.Iterator lines = text.iterator();
			while(lines.hasNext()) {
				String line = (String) lines.next();
				StringTokenizer tokens = new StringTokenizer(line,
					"" + HUMAN + COMPUTER);
				while(tokens.hasMoreTokens()) {
					ret.append(tokens.nextToken());
				}
				if(lines.hasNext()) ret.append("\n");
			}
			ret.append(buffer);
			return ret.toString();
		}

		//
		// methods as AdjustmentListener
		//
		public void adjustmentValueChanged(AdjustmentEvent e) {
			canvas.top_row = e.getValue();
			canvas.repaint();
		}

		//
		// methods as KeyListener only
		//
		public void keyPressed(KeyEvent e) { }
		public void keyReleased(KeyEvent e) { }
		public synchronized void keyTyped(KeyEvent e) {
			char c = e.getKeyChar();
			if(c == 8 || c == 0177 || e.paramString().indexOf("Backspace") >= 0) {
				if(buffer.length() > 0) {
					buffer = buffer.substring(0, buffer.length() - 1);
					scrollbar.update();
					moveToBottom();
					canvas.repaint();
				}
				return;
			}

			buffer = buffer + c;
			scrollbar.update();
			moveToBottom();
			canvas.repaint();
			if(c == '\n') {
				scrollbar.update();
				notifyAll();
			}
		}

		//
		// methods for manipulating text
		//
		protected void addText(char user, String to_add) {
			StringTokenizer lines = new StringTokenizer(to_add, "\n", true);
			while(lines.hasMoreTokens()) {
				String line = lines.nextToken();
				if(line.equals("\n")) {
					text.add("");
				} else {
					ensureUser(user);
					String current = (String) text.lastElement();
					text.setElementAt(current + line, text.size() - 1);
				}
			}
			scrollbar.update();
			canvas.repaint();
		}
		protected void ensureUser(char user) {
			String line;
			if(text.size() == 0) {
				line = "";
				text.add(line);
			} else {
				line = (String) text.lastElement();
			}

			int computer = line.lastIndexOf(COMPUTER);
			int human = line.lastIndexOf(HUMAN);
			char current = human > computer ? HUMAN : COMPUTER;
			if(current != user) {
				text.setElementAt(line + user, text.size() - 1);
			}
		}
		protected static int countOccurrences(String text, char what) {
			int count = 0;
			int pos = 0;
			while(true) {
				pos = text.indexOf(what, pos);
				if(pos < 0) return count;
				++count;
				++pos;
			}
		}
	}



	private static class MyFrame extends JFrame
			implements WindowListener, ActionListener {
		protected class ClipOwner implements ClipboardOwner {
			public void lostOwnership(Clipboard clipboard, Transferable contents) { }
		}
		
		protected ClipOwner owner = new ClipOwner();
		public JMenuItem clear_item = new JMenuItem("Clear");
		public JMenuItem copy_item  = new JMenuItem("Copy All");
		public JMenuItem close_item = new JMenuItem("Quit");
		// public JMenuItem close_item = new JMenuItem("Close");
		public IOArea io = new IOArea();
	
		public MyFrame(String title) {
			super(title);
			addWindowListener(this);
			setBackground(java.awt.Color.white);

			JMenuBar menubar = new JMenuBar();  setJMenuBar(menubar);
			JMenu menu = new JMenu("File");     menubar.add(menu);
			clear_item.addActionListener(this); menu.add(clear_item);
			copy_item.addActionListener(this);  menu.add(copy_item);
			close_item.addActionListener(this); menu.add(close_item);

			getContentPane().add(io);
			pack();
			io.grabFocus();
		}

		// ActionLister methods
		public void actionPerformed(ActionEvent e) {
			if(e.getSource() == clear_item) {
				io.clear();
			} else if(e.getSource() == copy_item) {
				Clipboard clipboard = frame.getToolkit().getSystemClipboard();
				StringSelection data = new StringSelection(io.getText());
				clipboard.setContents(data, owner);
			} else if(e.getSource() == close_item) {
				getFrame().dispose();
				frame = null;
			}
		}

		// WindowListener methods
		public void windowActivated(WindowEvent e) { }
		public void windowClosed(WindowEvent e) {
			if(DR_JAVA) {
				frame = null;
				buffer = new InputBuffer();
			} else {
				System.exit(0);
			}
		}
		public void windowClosing(WindowEvent e) {
			this.dispose();
		}
		public void windowDeactivated(WindowEvent e) { }
		public void windowDeiconified(WindowEvent e) { }
		public void windowIconified(WindowEvent e) { }
		public void windowOpened(WindowEvent e) { }
  	}

	private static class InputBuffer {
		public static final String skipchars = " \n\r\t";

		private String buf = null;

		private void getBuffer() {
			if(buf == null) buf = getFrame().io.readLine();
		}

		private void toNextWord() {
			int[] pos = new int[skipchars.length()];
			if(buf != null) {
				// find first occurrence of character not in skipchars
				int i;
				for(i = 0; i < buf.length(); i++) {
					if(skipchars.indexOf(buf.charAt(i)) < 0) break;
				}

				// trim off everything up to this character
				buf = buf.substring(i);
			}
			while(buf == null || buf.equals("")) {
				buf = null;
				getBuffer();

				// find first occurrence of character not in skipchars
				int i;
				for(i = 0; i < buf.length(); i++) {
					if(skipchars.indexOf(buf.charAt(i)) < 0) break;
				}

				// trim off everything up to this character
				buf = buf.substring(i);
			}
		}

		public String getLine() {
			if(buf != null && buf.equals("")) buf = null;
			getBuffer();
			String ret = buf;
			buf = null;
			return ret;
		}

		public char getChar() {
			getBuffer();
			if(buf.equals("")) {
				buf = null;
				return '\n';
			} else {
				char ret = buf.charAt(0);
				buf = buf.substring(1);
				return ret;
			}
		}

		public String getWord() {
			toNextWord();

			int pos = 0;
			while(pos < buf.length()
					&& skipchars.indexOf(buf.charAt(pos)) < 0) {
				++pos;
			}

			String ret = buf.substring(0, pos);
			buf = buf.substring(pos);
			return ret;
		}

		public boolean getBoolean() {
			return getWord().equalsIgnoreCase("true");
		}

		public int getInt() {
			toNextWord();

			int pos = 0;
			if(pos < buf.length() + 1 && buf.charAt(pos) == '-'
					&& Character.isDigit(buf.charAt(pos + 1))) {
				++pos;
			}
			while(pos < buf.length()
					&& Character.isDigit(buf.charAt(pos))) {
				++pos;
			}

			if(pos == 0) {
				buf = buf.substring(1);
				return Integer.MIN_VALUE;
			} else {
				int ret = Integer.parseInt(buf.substring(0, pos));
				buf = buf.substring(pos);
				return ret;
			}
		}

		public double getDouble() {
			toNextWord();

			int last = -1;
			int state = 0;
			for(int pos = 0; pos < buf.length() && state >= 0; pos++) {
				char c = buf.charAt(pos);
				int oldstate = state;
				state = -1;
				switch(oldstate) {
				case 0:
					if(Character.isDigit(c))      state = 2;
					else if(c == '+' || c == '-') state = 1;
					else if(c == '.')             state = 3;
					break;
				case 1:
					if(Character.isDigit(c))      state = 2;
					else if(c == '.')             state = 3;
					break;
				case 2:
					if(Character.isDigit(c))      state = 2;
					else if(c == '.')             state = 4;
					else if(c == 'e' || c == 'E') state = 5;
					break;
				case 3:
					if(Character.isDigit(c))      state = 4;
					break;
				case 4:
					if(Character.isDigit(c))      state = 4;
					else if(c == 'e' || c == 'E') state = 5;
					break;
				case 5:
					if(Character.isDigit(c))      state = 7;
					else if(c == '-' || c == '+') state = 6;
					break;
				case 6:
					if(Character.isDigit(c))      state = 7;
					break;
				case 7:
					if(Character.isDigit(c))      state = 7;
					break;
				}
				if(state == 2 || state == 4 || state == 7) {
					last = pos + 1;
				}
			}

			if(last < 0) {
				buf = buf.substring(1);
				return Double.NaN;
			} else {
				double ret = Double.parseDouble(buf.substring(0, last));
				buf = buf.substring(last);
				return ret;
			}
		}
	}

	private static MyFrame getFrame() {
		if(frame == null) {
			frame = new MyFrame("I/O Console");
			frame.pack();
			frame.show();
		}
		return frame;
	}

	/**
	 * Sets the title appearing in the title bar.
	 */
	public static void setTitle(String title) {
		getFrame().setTitle(title);
	}

	/**
	 * Resizes the window so that it can hold cols columns
	 * and rows rows of text.
	 */
	public static void setSize(int cols, int rows) {
		getFrame().io.setSize(cols, rows);
		getFrame().pack();
	}

	/**
	 * Removes the window from the screen.
	 */
	public static void dispose() {
		if(frame != null) {
			getFrame().dispose();
			frame = null;
		}
	}

	/**
	 * Clears all text in the window.
	 */
	public static void clear() {
		getFrame().io.clear();
	}

	/**
	 * Prints the object's value indicated into the window,
	 * without terminating the current line of output.
	 */
	public static void print(Object text) {
		getFrame().io.print(text.toString());
	}

	/**
	 * Prints the integer value indicated into the window,
	 * without terminating the current line of output.
	 */
	public static void print(int i) { print(Integer.toString(i)); }

	/**
	 * Prints the double-precision value indicated into the window,
	 * without terminating the current line of output.
	 */
	public static void print(double i) { print(Double.toString(i)); }

	/**
	 * Prints the Boolean value indicated into the window,
	 * without terminating the current line of output.
	 */
	public static void print(boolean i) { print("" + i); }

	/**
	 * Prints the character value indicated into the window,
	 * without terminating the current line of output.
	 */
	public static void print(char i) { print("" + i); }

	/**
	 * Terminates the current line of output.
	 */
	public static void println() {
		getFrame().io.print("\n");
	}

	/**
	 * Prints the object's value indicated, then terminates
	 * the current line of output.
	 */
	public static void println(Object text) {
		getFrame().io.print(text.toString() + '\n');
	}

	/**
	 * Prints the integer value indicated, then terminates
	 * the current line of output.
	 */
	public static void println(int i) { println(Integer.toString(i)); }

	/**
	 * Prints the double-precision value indicated, then terminates
	 * the current line of output.
	 */
	public static void println(double i) { println(Double.toString(i)); }

	/**
	 * Prints the Boolean value indicated, then terminates
	 * the current line of output.
	 */
	public static void println(boolean i) { println("" + i); }

	/**
	 * Prints the character value indicated, then terminates
	 * the current line of output.
	 */
	public static void println(char i) { println("" + i); }

	/**
	 * Returns the next line
	 * the user types into the window. Any terminating
	 * end-of-line characters are omitted.
	 */
	public static String readLine() {
		if(DR_JAVA) {
			getFrame().io.print("IO.readLine() is unavailable.\n");
			return "";
		}
		return buffer.getLine();
	}

	/**
	 * Returns the next
	 * character the user types into the window. This will be
	 * '\n' when the user reaches the end of a line.
	 */
	public static char readChar() {
		if(DR_JAVA) {
			getFrame().io.print("IO.readChar() is unavailable.\n");
			return ' ';
		}
		return buffer.getChar();
	}

	/**
	 * Returns the next
	 * integer the user types into the window, skipping any
	 * whitespace characters. If the next non-whitespace letters don't
	 * appear to represent an integer, it eats a single
	 * non-whitespace character and returns Double.NaN.
	 */
	public static int readInt() {
		if(DR_JAVA) {
			getFrame().io.print("IO.readInt() is unavailable.\n");
			return Integer.MIN_VALUE;
		}
		return buffer.getInt();
	}

	/**
	 * Returns the next
	 * real number the user types into the window, skipping any
	 * whitespace characters.
	 * It supports both regular doubles (like 6.1 or -4)
	 * and scientific notation (like 3e8).
	 * If the next non-whitespace characters
	 * don't appear to represent a real number, it eats a single
	 * non-whitespace character and returns Double.NaN.
	 */
	public static double readDouble() {
		if(DR_JAVA) {
			getFrame().io.print("IO.readDouble() is unavailable.\n");
			return Double.NaN;
		}
		return buffer.getDouble();
	}

	/**
	 * Returns the next
	 * Boolean value the user types into the window, skipping any
	 * whitespace characters. This will be true if the user
	 * types the word ``true'' (case is insignificant here).
	 * If the next non-whitespace letters don't appear to represent
	 * a Boolean, it return false.
	 */
	public static boolean readBoolean() {
		if(DR_JAVA) {
			getFrame().io.print("IO.readBoolean() is unavailable.\n");
			return false;
		}
		return buffer.getBoolean();
	}

	/**
	 * Returns the next
	 * string of non-whitespace characters the user types into the
	 * window.
	 */
	public static String readString() {
		if(DR_JAVA) {
			getFrame().io.print("IO.readString() is unavailable.\n");
			return "";
		}
		return buffer.getWord();
	}

}
