package csbsju.cs160;

/**
 * Represents a pen located in a drawing area. This pen tracks several
 * attributes: the pen's location, direction, and whether it is
 * currently down. When down, it draws lines as it moves.</p>
 *
 * <p>The following code example demonstrates how you might write a
 * program to draw a square using the Pen class.
 * <blockquote><pre>
 * import csbsju.cs160.gui.*;
 *
 * public class DrawSquare {
 *     public static void main(String[] args) {
 *         DrawableFrame frame = new DrawableFrame();
 *         Pen pen = frame.getPen(50.0, 50.0);
 *         pen.forward(100.0);
 *         pen.turnRight(90.0);
 *         pen.forward(100.0);
 *         pen.turnRight(90.0);
 *         pen.forward(100.0);
 *         pen.turnRight(90.0);
 *         pen.forward(100.0);
 *         frame.show();
 *     }
 * }
 * </pre></blockquote></p>
 *
 * <p>It may strike you as odd that the pen works in terms of
 * <tt>double</tt>s, even though graphics are drawn strictly in terms of
 * <tt>integer</tt> coordinates. This is so that the pen can move more
 * precisely - since, when the pen is moving obliquely to the axes, its
 * movement will rarely follow integer coordinates exactly.</p>
 *
 * @author  Carl Burch
 * @version 28 August 2001
 */
public class Pen {
	private Graphics g;
	private Color color = Color.black;
	private boolean pen_down = true;
	private double cur_x = 0.0;
	private double cur_y = 0.0;
	private double angle = 0.0;

	/** Creates a pen starting at location
	 * (<code>x</code>, <code>y</code>) heading east,
	 * using <code>g</code> to draw any future lines. */
	public Pen(Graphics g, double x, double y) {
		this.g = g;
		cur_x = x;
		cur_y = y;
	}

	/** Sets the color of the pen's future lines to
	 * <code>color</code>. (The default when the pen is created is
	 * black.) */
	public void setColor(Color color) {
		this.color = color;
	}
	/** Moves the pen to (<code>x</code>, <code>y</code>), without
	 * drawing a line. */
	public void moveTo(double x, double y) {
		cur_x = x;
		cur_y = y;
	}
	/** Moves the pen to (<code>x</code>, <code>y</code>),
	 * drawing a line from its previous location. */
	public void lineTo(double x, double y) {
		g.setColor(color);
		g.drawLine(round(cur_x), round(cur_y), round(x), round(y));
		cur_x = x;
		cur_y = y;
	}
	/** Moves the pen forward <code>dist</code> pixels in its current
	 * direction from its current location, tracing a line if the pen is
	 * currently down. */
	public void forward(double dist) {
		if(pen_down) {
			lineTo(cur_x + dist * Math.cos(angle), cur_y - dist * Math.sin(angle));
		} else {
			moveTo(cur_x + dist * Math.cos(angle), cur_y - dist * Math.sin(angle));
		}
	}
	/** Turns the pen's direction <code>degrees</code> degrees
	 * clockwise. */
	public void turnLeft(double degrees) {
		angle += degrees * Math.PI / 180.0;
	}
	/** Turns the pen's direction <code>degrees</code> degrees
	 * counterclockwise. */
	public void turnRight(double degrees) {
		angle -= degrees * Math.PI / 180.0;
	}
	/** Raises the pen, so that future calls to <code>forward()</tt> do
	 * not draw a line. */
	public void raisePen() {
		pen_down = false;
	}
	/** Lowers the pen, so that future calls to <code>forward()</tt> do
	 * will draw a line. (This is the initial behavior when the Pen is
	 * created.) */
	public void lowerPen() {
		pen_down = true;
	}

	private int round(double x) {
		return (int) (x + 0.5);
	}
}
