/*
 * Copyright (c) 2005, Carl Burch.
 * 
 * This file is part of the com.cburch.editor package. The latest
 * version is available at http://www.cburch.com/proj/editor/.
 *
 * The com.cburch.editor package is free software; you can redistribute
 * it and/or modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * The com.cburch.editor package is distributed in the hope that it will
 * be useful, but WITHOUT ANY WARRANTY; without even the implied warranty
 * of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with the com.cburch.editor package; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301  USA
 */
 
 package com.cburch.editor;

import java.awt.event.ActionEvent;
import java.awt.event.KeyEvent;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import javax.swing.AbstractAction;
import javax.swing.Action;
import javax.swing.ActionMap;
import javax.swing.InputMap;
import javax.swing.KeyStroke;
import javax.swing.text.BadLocationException;
import javax.swing.text.Caret;
import javax.swing.text.Document;
import javax.swing.text.JTextComponent;

import com.cburch.editor.tokens.TokenList;

/**
 * Handles the automatic indentation as the user types.
 * 
 * @author Carl Burch
 * @version 0.1 2005-05-31
 */
public class AutoIndent {
    /** A pattern used for identifying the whitespace at the
     * beginning of a line. */
    private static final Pattern whitespacePattern = Pattern.compile("\\s+");

    /** The action used whenever the user presses the enter key. */
    private class EnterAction extends AbstractAction {
        /* Constructs the action. */
        EnterAction() {
            super("insertNewlineIndent");
        }
        
        /* Performs the action of inserting the newline and the
         * indentation for the next line. */
        public void actionPerformed(ActionEvent event) {
            if(!enabled) oldCloseAction.actionPerformed(event);

            // remove current selection
            JTextComponent source = (JTextComponent) event.getSource();
            source.replaceSelection("");
            
            // determine caret location
            Caret caret = source.getCaret();
            if(caret == null) { return; }
            int loc = caret.getDot();
            if(loc < 0) { return; }
            
            try {
                // determine the indentation of the current line
                Document doc = source.getDocument();
                int lineStart = getLineStart(doc, loc);
                String line = doc.getText(lineStart, loc - lineStart);
                Matcher match = whitespacePattern.matcher(line);
                String init = match.lookingAt() ? match.group() : "";
                
                // add a tab if the line has an extra brace
                if(hasOpeningBrace(lineStart + init.length(), loc)) {
                    init = init + "    ";
                }
                
                // finally insert the newline and the indentation
                doc.insertString(loc, "\n" + init, null);
            } catch(BadLocationException e) {
                System.err.println("EnterAction: at bad location " + loc);
            }
        }
    }

    /** The action performed when the user closes a brace. */
    private class CloseAction extends AbstractAction {
        /** Constructs the closing-brace action. */
        CloseAction() {
            super("braceClose");
        }
        
        /** Performs the action of possibly unindenting if there is
         * only whitespace previous to this brace, and of course
         * inserts the brace in any case. */
        public void actionPerformed(ActionEvent event) {
            if(!enabled) oldCloseAction.actionPerformed(event);

            // remove current selection
            JTextComponent source = (JTextComponent) event.getSource();
            source.replaceSelection("");
            
            // determine caret location
            Caret caret = source.getCaret();
            if(caret == null) { return; }
            int loc = caret.getDot();
            if(loc < 0) { return; }
            
            Document doc = source.getDocument();
            int lineStart = getLineStart(doc, loc);
            try {
                // See whether there is only whitespace previous
                // to this brace - which makes us eligible for
                // unindentation.
                String line = doc.getText(lineStart, loc - lineStart);
                Matcher match = whitespacePattern.matcher(line);
                if(!match.matches()) {
                    // If it's not entirely white, then we don't
                    // have any unindentation to do. Just insert
                    // the closing brace and go.
                    doc.insertString(loc, "" + closeBraceChar, null);
                    return;
                }
                
                // Now fetch the whitespace on the previous line:
                // We don't want to unindent further than this.
                int prevStart = getLineStart(doc, lineStart - 1);
                String prevLine = doc.getText(prevStart, lineStart - prevStart);
                Matcher prevMatch = whitespacePattern.matcher(prevLine);
                String prevInit = prevMatch.lookingAt() ? prevMatch.group() : "";
                int offs = prevInit.length();
                
                // If there is not an opening brace on the
                // previous line, we can move back a little more.
                if(!hasOpeningBrace(prevStart + offs, lineStart)) {
                    int counter = 0;
                    while(offs > 0 && counter < 4) {
                        offs--;
                        char c = prevInit.charAt(offs);
                        if(c == '\t') counter = 4;
                        else counter++;
                    }
                }
                
                // Insert the brace and possibly unindent.
                doc.insertString(loc, "" + closeBraceChar, null);
                if(offs < line.length()) {
                    doc.remove(lineStart + offs, line.length() - offs);
                }
            } catch(BadLocationException e) {
                System.err.println("BraceCloseAction: at bad location " + loc);
            }
        }
    }

    /** The editor for which we should perform the indentation. */
    private Editor editor;
    
    /** The token we use for identifying an opening brace. */
    private TokenType lbrace;

    /** The token we use for identifying a closing brace. */
    private TokenType rbrace;
    
    /** The character used for the closing brace. */
    private char closeBraceChar;
    
    
    /** The action associated with the enter key before we change it.
     * We want it so that we can fall back when auto-indent is disabled. */
    private Action oldEnterAction;
    
    /** The action associated with the close-brace key before we change it.
     * We want it so that we can fall back when auto-indent is disabled. */
    private Action oldCloseAction;
    
    /** Whether automatic indentation is enabled. */
    private boolean enabled = false;
    
    /**
     * Constructs automatic indentation for the given editor.
     * 
     * @param editor  the editor for which we will perform automatic indentation.
     * @param openBrace  the token we use for identifying opening braces.
     * @param closeBrace  the token we use for identifying closing braces.
     * @param closeBraceChar  the character corresponding to closing braces.
     */
    public AutoIndent(Editor editor, TokenType openBrace,
            TokenType closeBrace, char closeBraceChar) {
        this.editor = editor;
        this.lbrace = openBrace;
        this.rbrace = closeBrace;
        this.closeBraceChar = closeBraceChar;
        
        KeyStroke enterKey = KeyStroke.getKeyStroke(KeyEvent.VK_ENTER, 0);
        KeyStroke closeKey = KeyStroke.getKeyStroke(closeBraceChar);

        InputMap inputMap = editor.getInputMap();
        ActionMap actionMap = editor.getActionMap();
        oldEnterAction = actionMap.get(inputMap.get(enterKey));
        oldCloseAction = actionMap.get(inputMap.get(closeKey));
        inputMap.put(enterKey, new EnterAction());
        inputMap.put(closeKey, new CloseAction());

        setEnabled(true);
    }

    /**
     * Returns <code>true</code> the automatic indentation is
     * currently enabled.
     * 
     * @return <code>true</code> if the indentation is enabled.
     */
    public boolean isEnabled() {
        return enabled;
    }
    
    /**
     * Controls whether the indentation takes place.
     * 
     * @param value  <code>true</code> if automatic indentation should
     *    take place, <code>false</code> if it should be disabled.
     */
    public void setEnabled(boolean value) {
        if(value == enabled) return;
        enabled = value;
    }

    /**
     * Determines whether there is an opening brace in a segment
     * of the token list.
     * 
     * @param start  the index at which to begin the search.
     * @param stop   the index at which to stop, exclusive.
     * @return  <code>true</code> if there is an opening brace,
     *    <code>false</code> if not.
     */
    private boolean hasOpeningBrace(int start, int stop) {
        TokenList<? extends Token> tokens
                = editor.getTokenizer().getTokenList();
        int index = tokens.getIndexStartingAfter(start);
        int count = 0;
        while(index < tokens.size()) {
            Token t = tokens.get(index);
            if(t.getEndOffset() > stop) break;
            if(t.getType() == lbrace) ++count;
            else if(t.getType() == rbrace && count > 0) --count;
            index++;
        }
        return count > 0;
    }

    /**
     * Determines the offset at the beginning of the line
     * containing the character with the given offset.
     * 
     * @param doc  the document whose offset we should find.
     * @param loc  the offset we are querying.
     * @return  the offset of the first character in the line
     *    containing <code>loc</code> also.
     */
    private static int getLineStart(Document doc, int loc) {
        while(true) {
            int cur = loc - 80;
            if(cur < 0) cur = 0;
            
            try {
                String text = doc.getText(cur, loc - cur);
                int ret = text.lastIndexOf("\n");
                if(ret >= 0) return cur + ret + 1;
                if(cur == 0) return 0;
            } catch(BadLocationException e) { return 0; }
            
            loc = cur;
        }
    }
}
