/*
 * Copyright (c) 2005, Carl Burch.
 * 
 * This file is part of the com.cburch.editor package. The latest
 * version is available at http://www.cburch.com/proj/editor/.
 *
 * The com.cburch.editor package is free software; you can redistribute
 * it and/or modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * The com.cburch.editor package is distributed in the hope that it will
 * be useful, but WITHOUT ANY WARRANTY; without even the implied warranty
 * of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with the com.cburch.editor package; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301  USA
 */
 
 package com.cburch.editor;

import java.awt.BasicStroke;
import java.awt.Color;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.Rectangle;
import java.awt.Shape;
import java.util.ArrayList;
import java.util.List;

import javax.swing.event.CaretEvent;
import javax.swing.event.CaretListener;
import javax.swing.text.BadLocationException;
import javax.swing.text.Document;
import javax.swing.text.Highlighter;
import javax.swing.text.JTextComponent;

import com.cburch.editor.tokens.MutableTokenList;
import com.cburch.editor.tokens.Tokenizer;
import com.cburch.editor.tokens.TokenList;
import com.cburch.editor.tokens.TokenizerEvent;
import com.cburch.editor.tokens.TokenizerListener;

/**
 * Handles parenthesis matching, both in highlighting unmatched
 * parentheses and in displaying the matches of the parenthesis
 * under the caret. (It works equally well for parentheses, braces,
 * and brackets. We can use the terms interchangably here.) 
 * 
 * @author Carl Burch
 * @version 0.1 2005-05-31
 */
public class BracketMatcher {
    /** If true, enables updates on changes to the bracket list. */ 
    private static final boolean DEBUG = false;
    
    /** The highlight used for pointing out unmatched parentheses. */
    private static final BoxHighlighter errorHighlight
        = new BoxHighlighter(new Color(255, 0, 0, 96));
    
    /** The highlight used for pointing out a parenthesis' match. */
    private static final BoxHighlighter matchHighlight
        = new BoxHighlighter(new Color(0, 150, 0, 96));

    /** Implements a highlighter for drawing a rectangle around
     * a single character. */
    private static class BoxHighlighter
            implements Highlighter.HighlightPainter {
        /** The color to draw this highlight with. */
        private Color color;
        
        /** Construct a highlighter for the given color. */
        public BoxHighlighter(Color color) {
            this.color = color;
        }
        
        /** Paints the highlight. We'll ignore the second index
         * because we are only highlighting a single index: The
         * highlight will grow with characters added just after the
         * highlight, which we do not want. */
        public void paint(Graphics g, int p0, int p1, Shape bounds,
                JTextComponent c) {
            if(p0 >= p1) return;
            try {
                Rectangle rect = c.modelToView(p0);
                Rectangle next = c.modelToView(p0 + 1);
                if(next.x < rect.x) return;
                rect.width = next.x - rect.x;
                g.setColor(color);
                if(g instanceof Graphics2D) {
                    ((Graphics2D) g).setStroke(new BasicStroke(3.0f));
                }
                g.drawRect(rect.x + 1, rect.y + 1,
                        rect.width - 2, rect.height - 2);
            } catch(BadLocationException e) { }
        }
    }
        
    /** A listener both to changes in the token sequence and to
     * the caret position. */
    private class MyListener implements TokenizerListener<Token>,
            CaretListener {
        /** When the list of tokens changes, we revise our own list
         * of brackets. */
        public void rangeReplaced(TokenizerEvent<Token> event) {
            Document doc = editor.getDocument();

            // determine the brackets that were added.
            MutableTokenList<Token> toAdd = new MutableTokenList<Token>(
                    fetchBrackets(event.getNewTokens()));
            
            // determine the first and last bracket removed.
            Token firstReplace = null;
            Token lastReplace = null;
            for(Token t : event.getOldTokens()) {
                TokenType type = t.getType();
                if(type == open || type == close) {
                    if(firstReplace == null) firstReplace = t;
                    lastReplace = t;
                    // TODO: We could potentially improve matters
                    // by checking to what extent the removed brackets
                    // correspond to the added brackets. 
                }
            }
            
            // find the indices of the first and last brackets
            // within our list.
            int startReplace;
            int stopReplace;
            if(firstReplace == null) {
                if(toAdd.size() == 0) return;
                startReplace = brackets.getIndexStartingAfter(toAdd.get(0).getEndPosition());
                stopReplace = startReplace;
            } else {
                /*
                startReplace = brackets.getIndexEndingBefore(firstReplace.getEndPosition());
                stopReplace = brackets.getIndexEndingBefore(lastReplace.getEndPosition()) + 1;
                */
                /* A more reliable O(n) version of the previous two lines: */
                int n = brackets.size();
                startReplace = 0;
                while(startReplace < n && brackets.get(startReplace) != firstReplace) startReplace++;
                stopReplace = startReplace;
                while(stopReplace < n && brackets.get(stopReplace) != lastReplace) stopReplace++;
                stopReplace++;
            }
            
            // Finally, update our list of brackets and recompute
            // the matches.
            if(DEBUG && (startReplace < stopReplace || toAdd.size() > 0)) {
                System.err.println("BracketMatcher: change made");
                for(int i = startReplace; i < stopReplace; i++) {
                    System.err.println("  removing " + brackets.get(i));
                }
                for(int i = 0; i < toAdd.size(); i++) {
                    System.err.println("  adding " + toAdd.get(i));
                }
            }
            brackets.replace(startReplace, stopReplace, toAdd);
            computeMatches();
        }

        /** Update the highlight showing the match to the bracket
         * adjacent to the caret. */
        public void caretUpdate(CaretEvent event) {
            // Remove the current matching highlight.
            if(matchTag != null) {
                highlighter.removeHighlight(matchTag);
            }

            // Determine which bracket is adjacent to caret.
            int offset = event.getDot();
            int index = brackets.getIndexEndingBefore(offset);
            if(index < 0 || index >= brackets.size()) return;
            Token selected = brackets.get(index);
            if(selected.getEndOffset() != offset) return;
            
            // Highlight the matching bracket, if it exists.
            Token match = (Token) selected.getData();
            if(match == null) return;
            try {
                matchTag = highlighter.addHighlight(match.getBeginOffset(), match.getEndOffset(), matchHighlight);
            } catch(BadLocationException e) {
                System.err.println("bad location");
            }
        }
    }
    
    /** The editor that we are highlighting. */
    private Editor editor;
    
    /** The highlighter used for drawing highlights in the editor. */
    private Highlighter highlighter;
    
    /** The token corresponding to an opening bracket. */
    private TokenType open;
    
    /** The token corresponding to a closing bracket. */
    private TokenType close;
    
    /** The error message associated with unmatched brackets. */
    private String errorMessage;

    /** A list of the brackets in the current document. */
    private MutableTokenList<Token> brackets;
    
    /** Whether the unmatched brackets should be highlighted. */
    private boolean unmatchedHighlighted = false;
    
    /** Whether the unmatched brackets should be highlighted. */
    private boolean caretMatchHighlighted = false;
    
    /** The listener we will use for listening to the tokenizer and
     * potentially the caret. */
    private MyListener myListener = new MyListener();
     
    /** The current highlights shown for unmatched brackets. */
    private ArrayList<Object> unmatchedTags = new ArrayList<Object>();
    
    /** The current highlight for the caret's matching bracket. */
    private Object matchTag = null;
    
    /**
     * Constructs a matcher for a particular editor.
     * 
     * @param editor  The editor we should listen to.
     * @param openType  The token category used for opening brackets.
     * @param closeType  The token category used for closing brackets.
     */
    public BracketMatcher(Editor editor, TokenType openType,
            TokenType closeType, String errorMessage) {
        this.editor = editor;
        this.highlighter = editor.getHighlighter();
        this.open = openType;
        this.close = closeType;
        this.errorMessage = errorMessage;
        setUnmatchedHighlighted(true);
        setCaretMatchHighlighted(true);
    }

    /**
     * Returns <code>true</code> if the editor should display
     * the bracket matching the bracket underneath the caret.
     * 
     * @return <code>true</code> if the matches are displayed.
     */
    public boolean isCaretMatchHighlighted() {
        return caretMatchHighlighted;
    }
    
    /**
     * Controls whether the matcher will display the bracket
     * matching the bracket underneath the caret.
     * 
     * @param value  <code>true</code> if the current match should be
     *     highlighted, <code>false</code> if it should not be.
     */
    public void setCaretMatchHighlighted(boolean value) {
        if(caretMatchHighlighted == value) return;
        caretMatchHighlighted = value;
        if(caretMatchHighlighted) {
            if(!unmatchedHighlighted) enableListener();
            editor.addCaretListener(myListener);
        } else {
            if(!unmatchedHighlighted) disableListener();
            if(matchTag != null) {
                highlighter.removeHighlight(matchTag);
                matchTag = null;
            }
            editor.removeCaretListener(myListener);
        }
    }

    /**
     * Returns <code>true</code> if unmatched brackets should be
     * highlighted as "errors."
     * 
     * @return <code>true</code> if the unmatched brackets are
     *     highlighted, <code>false</code> if they are not.
     */
    public boolean isUnmatchedHighlighted() {
        return unmatchedHighlighted;
    }
    
    /**
     * Controls whether the matcher will highlight unmatched
     * brackets.
     * 
     * @param value  <code>true</code> if the matcher should
     *     highlight unmatched brackets, <code>false</code> if
     *     it should not.
     */
    public void setUnmatchedHighlighted(boolean value) {
        if(unmatchedHighlighted == value) return;
        unmatchedHighlighted = value;
        if(unmatchedHighlighted) {
            if(!caretMatchHighlighted) enableListener();
            else highlightAllUnmatched();
        } else {
            if(!caretMatchHighlighted) disableListener();
            removeUnmatchedHighlights();
        }
    }

    /** Disables listening for changes to the token list. */
    private void disableListener() {
        Tokenizer<? extends Token> tokenizer = editor.getTokenizer();
        tokenizer.removeTokenizerListener(myListener);
    }

    /** Enables listening for changes to the token list. */
    private void enableListener() {
        Tokenizer<? extends Token> tokenizer = editor.getTokenizer();
        TokenList<? extends Token> tokenList = tokenizer.getTokenList();
        brackets = new MutableTokenList<Token>(fetchBrackets(tokenList));
        computeMatches();
        tokenizer.addTokenizerListener(myListener, true);
    }
    
    /** Fetch all brackets from the given list. */
    private ArrayList<Token> fetchBrackets(List<? extends Token> source) {
        ArrayList<Token> ret = null;
        for(int i = 0, n = source.size(); i < n; i++) {
            Token t = source.get(i);
            TokenType type = t.getType();
            if(type == open || type == close) {
                if(ret == null) ret = new ArrayList<Token>();
                ret.add(t);
            }
        }
        return ret;
    }

    /** Compute all the matches within the current list of brackets,
     * and highlight any unmatched brackets. */
    private void computeMatches() {
        removeUnmatchedHighlights();
        
        // recompute matches, adding error highlights as we go
        ArrayList<Token> stack = new ArrayList<Token>();
        for(int i = 0, n = brackets.size(); i < n; i++) {
            Token t = brackets.get(i);
            TokenType tType = t.getType();
            if(tType == open) {
                stack.add(t);
            } else if(tType == close) {
                int k = stack.size() - 1;
                if(k < 0) {
                    t.setData(null);
                    t.setErrorMessage(errorMessage);
                    highlightUnmatched(t);
                } else {
                    Token match = stack.remove(k);
                    match.setData(t);
                    t.setData(match);
                    match.setErrorMessage(null);
                    t.setErrorMessage(null);
                }
            }
            if(DEBUG) {
                System.err.println("  depth " + stack.size() + " after " + t);
            }
        }
        
        // Anything left on our stack is also unmatched.
        for(Token t : stack) {
            t.setData(null);
            t.setErrorMessage(errorMessage);
            highlightUnmatched(t);
        }
    }

    /** Remove all highlights for unmatched brackets. */ 
    private void removeUnmatchedHighlights() {
        for(Object tag : unmatchedTags) {
            highlighter.removeHighlight(tag);
        }
        unmatchedTags.clear();
    }

    /** Highlights all unmatched brackets. */
    private void highlightAllUnmatched() {
        for(Token t : brackets) {
            if(t.getData() == null) highlightUnmatched(t);
        }
    }
    
    /** Highlights a single unmatched bracket. */
    private void highlightUnmatched(Token t) {
        if(!unmatchedHighlighted) return;
        try {
            Object tag = highlighter.addHighlight(t.getBeginOffset(), t.getEndOffset(), errorHighlight);
            unmatchedTags.add(tag);
        } catch(BadLocationException e) {
            System.err.println("bad location");
        }
    }
}
