/*
 * Copyright (c) 2005, Carl Burch.
 * 
 * This file is part of the com.cburch.editor package. The latest
 * version is available at http://www.cburch.com/proj/editor/.
 *
 * The com.cburch.editor package is free software; you can redistribute
 * it and/or modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * The com.cburch.editor package is distributed in the hope that it will
 * be useful, but WITHOUT ANY WARRANTY; without even the implied warranty
 * of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with the com.cburch.editor package; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301  USA
 */
 
 package com.cburch.editor;

import java.beans.PropertyChangeListener;
import java.beans.PropertyChangeSupport;

import javax.swing.text.Document;
import javax.swing.event.DocumentEvent;
import javax.swing.event.DocumentListener;

/**
 * Tracks whether there have been any changes to a document.
 * 
 * @author Carl Burch
 * @version 0.1.1 2005-07-05
 */
public class DirtyTracker {
    /** The property associated with the dirtiness currently tracked. */
    public static final String isDirtyProperty = "isDirty";
    
    /** A listener that simply sets <code>dirty</code> to true for
     * any change to the document. */
    private class MyListener implements DocumentListener {
        public void insertUpdate(DocumentEvent arg0) { setDirty(true); } 
        public void removeUpdate(DocumentEvent arg0) { setDirty(true); }
        public void changedUpdate(DocumentEvent arg0) { }
    }
    
    /** The document whose "dirtyness" we are tracking. */
    private Document document;
    
    /** The listener we are using to track it. */
    private MyListener myListener = new MyListener();
    
    /** Tracks whether the document is currently dirty. */
    private boolean dirty = false;
    
    /** Tracks whether the tracking is currently enabled. */
    private boolean enabled = false;
    
    /** The listeners for property changes. */
    private PropertyChangeSupport propertySupport
        = new PropertyChangeSupport(this);

    /**
     * Constructs a tracker to determine when the document has
     * changed.
     * 
     * @param document  the document we are to track.
     */
    public DirtyTracker(Document document) {
        this.document = document;
        setEnabled(true);
    }
    
    /**
     * Returns <code>true</code> if the document has changed
     * since <code>setDirty(false)</code> was last called.
     * 
     * @return <code>true</code> if the document has changed.
     */
    public boolean isDirty() {
        return dirty;
    }
    
    /**
     * Sets the "dirtiness" state we use for the document. This
     * is most often used with an argument of <code>false</code>
     * (perhaps when the user has saved the text).
     * 
     * @param value  the value to assign to the "dirtiness"
     */
    public void setDirty(boolean value) {
        if(dirty == value) return;
        dirty = value;
        propertySupport.firePropertyChange(isDirtyProperty, !value, value);
    }
    
    /**
     * Sets the document that we are tracking. This has no effect
     * on whether this object thinks of the document as dirty; it
     * maintains what we had before.
     * 
     * @param value  the document to track.
     */
    public void setDocument(Document value) {
        boolean wasEnabled = enabled;
        setEnabled(false);
        document = value;
        if(wasEnabled) setEnabled(true);
    }
    
    /**
     * Returns <code>true</code> if this tracker is currently
     * enabled. It is enabled by default.
     * 
     * @return <code>true</code> if the tracker is currently
     *   tracking the document for changes.
     */
    public boolean isEnabled() {
        return enabled;
    }
    
    /**
     * Enables or disables the tracking. This has no effect on what
     * this object thinks of the document's "dirtiness."
     * 
     * @param value  <code>true</code> to enable this tracker,
     *     <code>false</code> to disable it.
     */
    public void setEnabled(boolean value) {
        if(value == enabled) return;
        enabled = value;
        if(enabled) {
            document.addDocumentListener(myListener);
        } else {
            document.removeDocumentListener(myListener);
        }
    }
    
    //
    // PropertyChangeSource methods
    //
    public void addPropertyChangeListener(PropertyChangeListener listener) {
        propertySupport.addPropertyChangeListener(listener);
    }
    public void addPropertyChangeListener(String propertyName, PropertyChangeListener listener) {
        propertySupport.addPropertyChangeListener(propertyName, listener);
    }
    public void removePropertyChangeListener(PropertyChangeListener listener) {
        propertySupport.removePropertyChangeListener(listener);
    }
    public void removePropertyChangeListener(String propertyName, PropertyChangeListener listener) {
        propertySupport.removePropertyChangeListener(propertyName, listener);
    }


}
