/*
 * Copyright (c) 2005, Carl Burch.
 * 
 * This file is part of the com.cburch.editor package. The latest
 * version is available at http://www.cburch.com/proj/editor/.
 *
 * The com.cburch.editor package is free software; you can redistribute
 * it and/or modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * The com.cburch.editor package is distributed in the hope that it will
 * be useful, but WITHOUT ANY WARRANTY; without even the implied warranty
 * of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with the com.cburch.editor package; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301  USA
 */
 
 package com.cburch.editor;

import java.io.IOException;
import java.io.Reader;

/**
 * The interface that any scanner associated with an editor
 * should implement.
 * 
 * @author Carl Burch
 * @version 0.1 2005-05-31
 *
 * @param <T> The type of token this scanner is to read.
 */
public interface Scanner<T> {
    /**
     * Resets the scanner to its state just after reading the
     * given token from the given reader. Normally, an implementation
     * would just use the token's <code>getEndOffset</code> method to
     * reset the current position; but depending on the scanner, the
     * token may hold other state to restore also.
     *
     * @param reader the new input.
     * @param token  the token after which we will start.
     * @throws IOException if an IOException occurs while switching
     *    readers.
     */
    public void reset(Reader reader, T token) throws IOException;

    /**
     * Closes the current input stream, and resets the scanner to read
     * from a new input stream. All internal variables are reset; the
     * old input stream cannot be reused. (The content of the internal
     * buffer is discarded and lost.) The lexical state is set to the
     * initial state.
     *
     * @param reader The new input.
     * @throws IOException if an IOExecption occurs while switching readers.
     */
    public void reset(Reader reader) throws IOException;
    
    /**
     * Returns the next token found on the input stream, or null
     * if there are no more tokens left.
     * 
     * @return the next token found on the input stream.
     * 
     * @throws IOException if the underlying Reader throws it during the process.
     */
    public T nextToken() throws IOException;
}
