/*
 * Copyright (c) 2005, Carl Burch.
 * 
 * This file is part of the com.cburch.editor package. The latest
 * version is available at http://www.cburch.com/proj/editor/.
 *
 * The com.cburch.editor package is free software; you can redistribute
 * it and/or modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * The com.cburch.editor package is distributed in the hope that it will
 * be useful, but WITHOUT ANY WARRANTY; without even the implied warranty
 * of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with the com.cburch.editor package; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301  USA
 */
 
 package com.cburch.editor.demo;

import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.InputEvent;
import java.awt.event.KeyEvent;

import javax.swing.JMenu;
import javax.swing.JMenuItem;
import javax.swing.JOptionPane;
import javax.swing.KeyStroke;
import javax.swing.event.CaretEvent;
import javax.swing.event.CaretListener;

import com.cburch.editor.EditorPanel;

/**
 * Implements the Edit menu.
 * 
 * @author Carl Burch
 * @version 0.1 2005-05-31
 */
class EditMenu extends JMenu {
    /** Listens for user selections of menu items. */
    private class MyListener implements ActionListener, CaretListener {
        public void actionPerformed(ActionEvent event) {
            Object source = event.getSource();
            if(source == cut) {
                editorPanel.getEditor().cut();
            } else if(source == copy) {
                editorPanel.getEditor().copy();
            } else if(source == paste) {
                editorPanel.getEditor().paste();
            } else if(source == clear) {
                editorPanel.getEditor().replaceSelection("");
            } else if(source == goLine) {
                goToLine();
            }
        }

        public void caretUpdate(CaretEvent event) {
            boolean selectionExists = event.getDot() != event.getMark();
            copy.setEnabled(selectionExists);
            cut.setEnabled(selectionExists);
            clear.setEnabled(selectionExists);
        }
    }
    
    private EditorPanel editorPanel;
    private MyListener myListener = new MyListener();
    private JMenuItem cut   = new JMenuItem("Cut");
    private JMenuItem copy  = new JMenuItem("Copy");
    private JMenuItem paste = new JMenuItem("Paste");
    private JMenuItem clear = new JMenuItem("Clear");
    private JMenuItem goLine = new JMenuItem("Go To Line");

    /**
     * Constructs the Edit menu.
     * 
     * @param editorPanel  the editor this menu is controlling.
     */
    public EditMenu(EditorPanel editorPanel) {
        super("Edit");
        this.editorPanel = editorPanel;
        
        add(cut);           cut.addActionListener(myListener);
        add(copy);          copy.addActionListener(myListener);
        add(paste);         paste.addActionListener(myListener);
        add(clear);         clear.addActionListener(myListener);
        addSeparator();
        add(goLine);        goLine.addActionListener(myListener);
        
        cut.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_X, InputEvent.CTRL_MASK));
        copy.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_C, InputEvent.CTRL_MASK));
        paste.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_V, InputEvent.CTRL_MASK));
        clear.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_DELETE, 0));
        goLine.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_L, InputEvent.CTRL_MASK));
        
        editorPanel.getEditor().addCaretListener(myListener);
    }
    
    /** Goes to a line that the user requests. */
    private void goToLine() {
        String lineNum = JOptionPane.showInputDialog(editorPanel,
                "Which line?", "Go To Line", JOptionPane.QUESTION_MESSAGE);
        if(lineNum == null) return;
        try {
            int line = Integer.parseInt(lineNum);
            int offset = editorPanel.getLineNumbers().toOffset(line);
            editorPanel.getEditor().setCaretPosition(offset);
        } catch(NumberFormatException e) {
            JOptionPane.showMessageDialog(editorPanel,
                    "The line number is not a valid integer",
                    "Invalid Line Number",
                    JOptionPane.INFORMATION_MESSAGE);
        }
    }
}
