/*
 * Copyright (c) 2005, Carl Burch.
 * 
 * This file is part of the com.cburch.editor package. The latest
 * version is available at http://www.cburch.com/proj/editor/.
 *
 * The com.cburch.editor package is free software; you can redistribute
 * it and/or modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * The com.cburch.editor package is distributed in the hope that it will
 * be useful, but WITHOUT ANY WARRANTY; without even the implied warranty
 * of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with the com.cburch.editor package; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301  USA
 */
 
 package com.cburch.editor.demo;

import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.InputEvent;
import java.awt.event.KeyEvent;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;

import javax.swing.JFileChooser;
import javax.swing.JMenu;
import javax.swing.JMenuItem;
import javax.swing.JOptionPane;
import javax.swing.KeyStroke;

import com.cburch.editor.DirtyTracker;
import com.cburch.editor.Editor;
import com.cburch.editor.EditorPanel;

/**
 * Implements the File menu.
 * 
 * @author Carl Burch
 * @version 0.1 2005-05-31
 */
class FileMenu extends JMenu {
    /** Listens for clicks on the menu items. */
    private class MyListener implements ActionListener {
        public void actionPerformed(ActionEvent event) {
            Object source = event.getSource();
            if(source == newi) {
                if(confirmClose("New")) {
                    editorPanel.getEditor().setText("");
                    curFile = null;
                    save.setEnabled(false);
                    tracker.setDirty(false);
                }
            } else if(source == open) {
                if(confirmClose("Open")) {
                    doOpen();
                }
            } else if(source == save) {
                doSave(false);
            } else if(source == saveAs) {
                doSave(true);
            } else if(source == quit) {
                if(confirmClose("Quit")) System.exit(0);
            }
        }
    }
    
    private EditorPanel editorPanel;
    private MyListener myListener = new MyListener();
    private JMenuItem newi   = new JMenuItem("New");
    private JMenuItem open   = new JMenuItem("Open...");
    private JMenuItem save   = new JMenuItem("Save");
    private JMenuItem saveAs = new JMenuItem("Save As...");
    private JMenuItem quit   = new JMenuItem("Quit");

    /** The current file being edited. */
    private File curFile = null;
    /** Tracks whether the file is consistent with what is on disk. */
    private DirtyTracker tracker;
    /** The file chooser to use when opening or saving files. */
    private JFileChooser chooser = null;

    /**
     * Constructs the File menu.
     * 
     * @param editorPanel  the editor controlled by this menu.
     */
    public FileMenu(EditorPanel editorPanel) {
        super("File");
        this.editorPanel = editorPanel;
        tracker = new DirtyTracker(editorPanel.getEditor().getDocument());
        
        add(newi);          newi.addActionListener(myListener);
        add(open);          open.addActionListener(myListener);
        add(save);          save.addActionListener(myListener);
        add(saveAs);        saveAs.addActionListener(myListener);
        addSeparator();
        add(quit);      quit.addActionListener(myListener);
        
        newi.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_N, InputEvent.CTRL_MASK));
        open.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_O, InputEvent.CTRL_MASK));
        save.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_S, InputEvent.CTRL_MASK));
        quit.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_Q, InputEvent.CTRL_MASK));
        
        save.setEnabled(false);
    }
    
    /**
     * Confirms whether the user really wants to switch from the
     * current file.
     * 
     * @param actionName  the name of the action requested, which
     *     is needed for creating a meaningful title.
     * @return <code>true</code> if the user is OK with continuing.
     */
    public boolean confirmClose(String actionName) {
        if(!tracker.isDirty()) return true;
        String[] options = { "Save", "Discard", "Cancel" };
        int result = JOptionPane.showOptionDialog(editorPanel,
                "What do you want to do with your unsaved changes?",
                "Confirm " + actionName, 0,
                JOptionPane.QUESTION_MESSAGE, null,
                options, options[0]);
        if(result == 0) { 
            return doSave(true);
        } else if(result == 1) { 
            return true;
        } else {
            return false;
        }
    }
    
    /**
     * Saves the file, showing a dialog to select the file if there
     * is no current file or if the argument is <code>true</code>;
     * otherwise it saves into the current file.
     * 
     * @param promptUser  indicates whether to prompt the user
     *     for the file name to be saved.
     * @return true  if the save is successful.
     */
    private boolean doSave(boolean promptUser) {
        File f = curFile;
        if(promptUser || f == null) {
            if(chooser == null) chooser = new JFileChooser();
            int action = chooser.showSaveDialog(editorPanel);
            if(action != JFileChooser.APPROVE_OPTION) return false;
            f = chooser.getSelectedFile();
            if(f == null) return false;
        }
        FileWriter out;
        try {
            out = new FileWriter(f);
        } catch(IOException e) {
            JOptionPane.showMessageDialog(editorPanel,
                    "Error opening file for saving.",
                    "Save Error", JOptionPane.ERROR_MESSAGE);
            return false;
        }
        try {
            out.write(editorPanel.getEditor().getText());
            out.close();
        } catch(IOException e) {
            JOptionPane.showMessageDialog(editorPanel,
                    "Save could not be completed.",
                    "Save Error", JOptionPane.ERROR_MESSAGE);
            return false;
        }
        curFile = f;
        save.setEnabled(true);
        tracker.setDirty(false);
        return true;
    }
    
    /**
     * Opens a file selected by the user.
     */
    private void doOpen() {
        // prompt the user
        if(chooser == null) chooser = new JFileChooser();
        int action = chooser.showOpenDialog(editorPanel);
        if(action != JFileChooser.APPROVE_OPTION) return;
        File f = chooser.getSelectedFile();
        
        // read the file in
        FileReader in;
        try {
            in = new FileReader(f); 
        } catch(FileNotFoundException e) {
            JOptionPane.showMessageDialog(editorPanel,
                    "File could not be found",
                    "Open Failed",
                    JOptionPane.ERROR_MESSAGE);
            return;
        }
        StringBuilder data = new StringBuilder();
        char[] buffer = new char[1024];
        try {
            while(true) {
                int count = in.read(buffer);
                if(count < 0) break;
                data.append(buffer, 0, count);
            }
            in.close();
        } catch(IOException e) {
            JOptionPane.showMessageDialog(editorPanel,
                    "Error while reading file",
                    "I/O Error", JOptionPane.ERROR_MESSAGE);
            return;
        }
        
        // now put it into the editor
        Editor editor = editorPanel.getEditor();
        editor.setText(data.toString());
        editor.setCaretPosition(0);
        curFile = f;
        save.setEnabled(true);
        tracker.setDirty(false);
    }
}
