/*
 * Copyright (c) 2005, Carl Burch.
 * 
 * This file is part of the com.cburch.editor package. The latest
 * version is available at http://www.cburch.com/proj/editor/.
 *
 * The com.cburch.editor package is free software; you can redistribute
 * it and/or modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * The com.cburch.editor package is distributed in the hope that it will
 * be useful, but WITHOUT ANY WARRANTY; without even the implied warranty
 * of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with the com.cburch.editor package; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301  USA
 */
 
 package com.cburch.editor.scanners;

import java.util.HashMap;

/**
 * Represents the information contained within a tag in an HTML
 * document.
 * 
 * @author Carl Burch
 * @version 0.1 2005-05-31
 */
public class HtmlTag {
    /** Represents information about an HTML 4.0 tag. */
    private static class KnownTag {
        /** The name of this tag. */
        String name;
        /** Whether we can have an ending tag of this type.
         * (You can't have &lt;/BR&gt; or &lt;/LI&gt;, for example. */
        boolean endTagOk;
        /** Whether this tag is deprecated in HTML 4.0. */
        boolean deprecated;
        /** Constructs the object with all information initialized. */
        KnownTag(String n, boolean e, boolean d) {
            name = n; endTagOk = e; deprecated = d;
        }
    }
    
    /** A structure containing all information about HTML 4.0 tags. */
    private static HashMap<String,KnownTag> knownTags = null;
    
    /** Initialize the structure if it is not already done. */
    private static void ensureTagsKnown() {
        if(knownTags != null) return;
        knownTags = new HashMap<String,KnownTag>();
        addTag("A", true, false);
        addTag("ABBR", true, false);
        addTag("ACRONYM", true, false);
        addTag("ADDRESS", true, false);
        addTag("APPLET", true, true);
        addTag("AREA", false, false);
        addTag("B", true, false);
        addTag("BASE", false, false);
        addTag("BASEFONT", false, true);
        addTag("BDO", true, false);
        addTag("BIG", true, false);
        addTag("BLOCKQUOTE", true, false);
        addTag("BODY", true, false);
        addTag("BR", false, false);
        addTag("BUTTON", true, false);
        addTag("CAPTION", true, false);
        addTag("CENTER", true, true);
        addTag("CITE", true, false);
        addTag("CODE", true, false);
        addTag("COL", false, false);
        addTag("COLGROUP", true, false);
        addTag("DD", true, false);
        addTag("DEL", true, false);
        addTag("DFN", true, false);
        addTag("DIR", true, true);
        addTag("DIV", true, false);
        addTag("DL", true, false);
        addTag("DT", true, false);
        addTag("EM", true, false);
        addTag("FIELDSET", true, false);
        addTag("FONT", true, true);
        addTag("FORM", true, false);
        addTag("FRAME", false, false);
        addTag("FRAMESET", true, false);
        addTag("H1", true, false);
        addTag("H2", true, false);
        addTag("H3", true, false);
        addTag("H4", true, false);
        addTag("H5", true, false);
        addTag("H6", true, false);
        addTag("HEAD", true, false);
        addTag("HR", false, false);
        addTag("HTML", true, false);
        addTag("I", true, false);
        addTag("IFRAME", true, false);
        addTag("IMG", false, false);
        addTag("INPUT", false, false);
        addTag("INS", true, false);
        addTag("ISINDEX", false, true);
        addTag("KBD", true, false);
        addTag("LABEL", true, false);
        addTag("LEGEND", true, false);
        addTag("LI", true, false);
        addTag("LINK", false, false);
        addTag("MAP", true, false);
        addTag("MENU", true, true);
        addTag("META", false, false);
        addTag("NOFRAMES", true, false);
        addTag("NOSCRIPT", true, false);
        addTag("OBJECT", true, false);
        addTag("OL", true, false);
        addTag("OPTGROUP", true, false);
        addTag("OPTION", true, false);
        addTag("P", true, false);
        addTag("PARAM", false, false);
        addTag("PRE", true, false);
        addTag("Q", true, false);
        addTag("S", true, true);
        addTag("SAMP", true, false);
        addTag("SCRIPT", true, false);
        addTag("SELECT", true, false);
        addTag("SMALL", true, false);
        addTag("SPAN", true, false);
        addTag("STRIKE", true, true);
        addTag("STRONG", true, false);
        addTag("STYLE", true, false);
        addTag("SUB", true, false);
        addTag("SUP", true, false);
        addTag("TABLE", true, false);
        addTag("TBODY", true, false);
        addTag("TD", true, false);
        addTag("TEXTAREA", true, false);
        addTag("TFOOT", true, false);
        addTag("TH", true, false);
        addTag("THEAD", true, false);
        addTag("TITLE", true, false);
        addTag("TR", true, false);
        addTag("TT", true, false);
        addTag("U", true, true);
        addTag("UL", true, false);
        addTag("VAR", true, false);
        addTag("A", true, false);
        addTag("ABBR", true, false);
        addTag("ACRONYM", true, false);
        addTag("ADDRESS", true, false);
        addTag("APPLET", true, true);
        addTag("AREA", false, false);
        addTag("B", true, false);
        addTag("BASE", false, false);
        addTag("BASEFONT", false, true);
        addTag("BDO", true, false);
        addTag("BIG", true, false);
        addTag("BLOCKQUOTE", true, false);
        addTag("BODY", true, false);
        addTag("BR", false, false);
        addTag("BUTTON", true, false);
        addTag("CAPTION", true, false);
        addTag("CENTER", true, true);
        addTag("CITE", true, false);
        addTag("CODE", true, false);
        addTag("COL", false, false);
        addTag("COLGROUP", true, false);
        addTag("DD", true, false);
        addTag("DEL", true, false);
        addTag("DFN", true, false);
        addTag("DIR", true, true);
        addTag("DIV", true, false);
        addTag("DL", true, false);
        addTag("DT", true, false);
        addTag("EM", true, false);
        addTag("FIELDSET", true, false);
        addTag("FONT", true, true);
        addTag("FORM", true, false);
        addTag("FRAME", false, false);
        addTag("FRAMESET", true, false);
        addTag("H1", true, false);
        addTag("H2", true, false);
        addTag("H3", true, false);
        addTag("H4", true, false);
        addTag("H5", true, false);
        addTag("H6", true, false);
        addTag("HEAD", true, false);
        addTag("HR", false, false);
        addTag("HTML", true, false);
        addTag("I", true, false);
        addTag("IFRAME", true, false);
        addTag("IMG", false, false);
        addTag("INPUT", false, false);
        addTag("INS", true, false);
        addTag("ISINDEX", false, true);
        addTag("KBD", true, false);
        addTag("LABEL", true, false);
        addTag("LEGEND", true, false);
        addTag("LI", true, false);
        addTag("LINK", false, false);
        addTag("MAP", true, false);
        addTag("MENU", true, true);
        addTag("META", false, false);
        addTag("NOFRAMES", true, false);
        addTag("NOSCRIPT", true, false);
        addTag("OBJECT", true, false);
        addTag("OL", true, false);
        addTag("OPTGROUP", true, false);
        addTag("OPTION", true, false);
        addTag("P", true, false);
        addTag("PARAM", false, false);
        addTag("PRE", true, false);
        addTag("Q", true, false);
        addTag("S", true, true);
        addTag("SAMP", true, false);
        addTag("SCRIPT", true, false);
        addTag("SELECT", true, false);
        addTag("SMALL", true, false);
        addTag("SPAN", true, false);
        addTag("STRIKE", true, true);
        addTag("STRONG", true, false);
        addTag("STYLE", true, false);
        addTag("SUB", true, false);
        addTag("SUP", true, false);
        addTag("TABLE", true, false);
        addTag("TBODY", true, false);
        addTag("TD", true, false);
        addTag("TEXTAREA", true, false);
        addTag("TFOOT", true, false);
        addTag("TH", true, false);
        addTag("THEAD", true, false);
        addTag("TITLE", true, false);
        addTag("TR", true, false);
        addTag("TT", true, false);
        addTag("U", true, true);
        addTag("UL", true, false);
        addTag("VAR", true, false);
    }

    /** A helper method for placing information into the structure. */
    private static void addTag(String name, boolean e, boolean d) {
        knownTags.put(name, new KnownTag(name, e, d));
    }
    
    /** Retrieve the information for a particular category of tag. */
    private static KnownTag getKnownTag(String name) {
        ensureTagsKnown();
        return knownTags.get(name);
    }

    /** Whether this tag starts its element. */
    private boolean startTag = true;
    /** Whether this tag ends its element. */
    private boolean endTag = false;
    /** The name of this tag. */
    private String name = null;
    /** The attribute values designated within the tag. */
    private HashMap<String,String> attrs = null;
    /** An error message associated with parsing the tag, if any. */
    private String errorMessage = null;

    //
    // The following package-protected methods are only for use
    // for the HtmlTagFactory method to initialize a tag object.
    //
    /** Constructs an empty HTML tag. */
    HtmlTag() { }
    
    /** Sets whether this tag starts its element. */
    void setStartTag(boolean value) { startTag = value; }
    /** Sets whether this tag ends its element. */
    void setEndTag(boolean value) { endTag = value; }
    /** Sets this tag's name. */
    void setName(String value) { name = value.toUpperCase(); }
    /** Designated that the attribute name exists. */
    void addAttribute(String attr) {
        if(attrs == null) attrs = new HashMap<String,String>();
        attrs.put(attr.toUpperCase(), null);
    }
    /** Associates a value with an attribute name. */
    void setAttribute(String attr, String value) {
        if(attrs == null) attrs = new HashMap<String,String>();
        attrs.put(attr.toUpperCase(), value);
    }
    /** Sets the error message associated with this tag. */
    void setError(String value) { if(errorMessage == null) errorMessage = value; }
    
    /**
     * Returns <code>true</code> if this tag starts its element.
     * 
     * @return <code>true</code> if the tag starts its element.
     */
    public boolean isStartTag() { return startTag; }
    
    /**
     * Returns <code>true</code> if this tag ends its element.
     * It is possible for a tag to both start and end the element if
     * it uses the XML-like syntax "<tt>&lt;TAG ... /&gt;</tt>".
     * 
     * @return <code>true</code> if this tag ends its element.
     */
    public boolean isEndTag() { return endTag; }
    
    /**
     * Returns the name within the tag, normalized to be in all
     * capital letters.
     * 
     * @return the name within the tag.
     */
    public String getName() { return name; }
    
    /**
     * Returns <code>true</code> if this tag mentions the requested
     * attribute.
     * 
     * @param name  the attribute name query.
     * @return <code>true</code> if the attribute is mentioned within
     *    the tag.
     */
    public boolean containsAttribute(String name) {
        return attrs == null ? false : attrs.containsKey(name.toUpperCase());
    }
    
    /**
     * Returns the value associated with the attribute, or <code>null</code>
     * if the attribute was not mentioned or was mentioned with no
     * corresponding value (as in "<tt>&lt;INPUT READONLY&gt;</tt>").
     *   
     * @param name  the attribute name query.
     * @return the value associated with the attribute name, or
     *    <code>null</code> if no value is associated.
     */
    public String getAttribute(String name) {
        return attrs == null ? null : attrs.get(name.toUpperCase());
    }
    
    /**
     * Returns an error message explaining any problems with the
     * HTML tag, or <code>null</code> if the HTML tag appears to
     * be valid.
     * 
     * @return an error message if the tag interpretation was not
     *    successful, or <code>null</code> if it was successful.
     */
    public String getError() {
        if(errorMessage != null) return errorMessage;
        if(name == null) return "Tag name is missing.";
        if(!isStartTag()) {
            KnownTag tagData = getKnownTag(name);
            if(tagData != null && !tagData.endTagOk) {
                return name + " end tags are not valid in HTML 4.0.";
            }
        }
        return null;
    }
    
    /**
     * Returns <code>true</code> if the HTML tag is deprecated in
     * HTML 4.01.
     * 
     * @return <code>true</code> if the tag is deprecated.
     */
    public boolean isDeprecated() {
        if(name == null) return false;
        KnownTag tagData = getKnownTag(name);
        if(tagData == null) return false;
        return tagData.deprecated;
    }
    
    /**
     * Returns <code>true</code> if the HTML tag is defined in the
     * HTML 4.01 standard.
     * 
     * @return <code>true</code> if the HTML tag is defined in the
     *     HTML 4.01 standard.
     */
    public boolean isKnown() {
        if(name == null) return false;
        KnownTag tagData = getKnownTag(name);
        return tagData != null;
    }
    
    /**
     * Generates a tag containing all the data encapsulated in the
     * given HTML string.
     * 
     * @param text  a string of HTML, including the enclosing angle
     *      brackets.
     * @return  an object containing all information encapsulated in
     *      the string.
     */
    public static HtmlTag create(String text) {
        return HtmlTagFactory.create(text);
    }
}