/*
 * Copyright (c) 2005, Carl Burch.
 * 
 * This file is part of the com.cburch.editor package. The latest
 * version is available at http://www.cburch.com/proj/editor/.
 *
 * The com.cburch.editor package is free software; you can redistribute
 * it and/or modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * The com.cburch.editor package is distributed in the hope that it will
 * be useful, but WITHOUT ANY WARRANTY; without even the implied warranty
 * of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with the com.cburch.editor package; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301  USA
 */
 
 package com.cburch.editor.scanners;

import java.util.HashMap;

import javax.swing.text.AttributeSet;

import com.cburch.editor.TokenType;

/**
 * Defines the different categories of tokens that might
 * be encountered in Java source code. You can find
 * additional documentation for all of these from
 * the <em><a href=http://java.sun.com/docs/books/jls/>Java
 * Language Specification</a></em>.
 * 
 * @author Carl Burch
 * @version 0.1 2005-05-31
 */
public enum JavaTokenType implements TokenType {
    /** The <code>abstract</code> keyword. */
    ABSTRACT("'abstract'", Styles.KEYWORD),
    /** The <code>assert</code> keyword, introduced in Java 1.4. */
    ASSERT("'assert'", Styles.KEYWORD),
    /** The <code>boolean</code> keyword. */
    BOOLEAN("'boolean'", Styles.KEYWORD),
    /** The <code>break</code> keyword. */
    BREAK("'break'", Styles.KEYWORD),
    /** The <code>byte</code> keyword. */
    BYTE("'byte'", Styles.KEYWORD),
    /** The <code>case</code> keyword. */
    CASE("'case'", Styles.KEYWORD),
    /** The <code>catch</code> keyword. */
    CATCH("'catch'", Styles.KEYWORD),
    /** The <code>char</code> keyword. */
    CHAR("'char'", Styles.KEYWORD),
    /** The <code>class</code> keyword. */
    CLASS("'class'", Styles.KEYWORD),
    /** The <code>const</code> keyword. */
    CONST("'const'", Styles.KEYWORD),
    /** The <code>continue</code> keyword. */
    CONTINUE("'continue'", Styles.KEYWORD),
    /** The <code>default</code> keyword. */
    DEFAULT("'default'", Styles.KEYWORD),
    /** The <code>do</code> keyword. */
    DO("'do'", Styles.KEYWORD),
    /** The <code>double</code> keyword. */
    DOUBLE("'double'", Styles.KEYWORD),
    /** The <code>else</code> keyword. */
    ELSE("'else'", Styles.KEYWORD),
    /** The <code>enum</code> keyword, introduced with Java 1.5. */
    ENUM("'enum'", Styles.KEYWORD),
    /** The <code>extends</code> keyword. */
    EXTENDS("'extends'", Styles.KEYWORD),
    /** The <code>final</code> keyword. */
    FINAL("'final'", Styles.KEYWORD),
    /** The <code>finally</code> keyword. */
    FINALLY("'finally'", Styles.KEYWORD),
    /** The <code>float</code> keyword. */
    FLOAT("'float'", Styles.KEYWORD),
    /** The <code>for</code> keyword. */
    FOR("'for'", Styles.KEYWORD),
    /** The <code>goto</code> keyword. */
    GOTO("'goto'", Styles.KEYWORD),
    /** The <code>if</code> keyword. */
    IF("'if'", Styles.KEYWORD),
    /** The <code>implements</code> keyword. */
    IMPLEMENTS("'implements'", Styles.KEYWORD),
    /** The <code>import</code> keyword. */
    IMPORT("'import'", Styles.KEYWORD),
    /** The <code>instanceof</code> keyword. */
    INSTANCEOF("'instanceof'", Styles.KEYWORD),
    /** The <code>int</code> keyword. */
    INT("'int'", Styles.KEYWORD),
    /** The <code>interface</code> keyword. */
    INTERFACE("'interface'", Styles.KEYWORD),
    /** The <code>long</code> keyword. */
    LONG("'long'", Styles.KEYWORD),
    /** The <code>native</code> keyword. */
    NATIVE("'native'", Styles.KEYWORD),
    /** The <code>new</code> keyword. */
    NEW("'new'", Styles.KEYWORD),
    /** The <code>package</code> keyword. */
    PACKAGE("'package'", Styles.KEYWORD),
    /** The <code>private</code> keyword. */
    PRIVATE("'private'", Styles.KEYWORD),
    /** The <code>protected</code> keyword. */
    PROTECTED("'protected'", Styles.KEYWORD),
    /** The <code>public</code> keyword. */
    PUBLIC("'public'", Styles.KEYWORD),
    /** The <code>return</code> keyword. */
    RETURN("'return'", Styles.KEYWORD),
    /** The <code>short</code> keyword. */
    SHORT("'short'", Styles.KEYWORD),
    /** The <code>static</code> keyword. */
    STATIC("'static'", Styles.KEYWORD),
    /** The <code>strictfp</code> keyword. */
    STRICTFP("'strictfp'", Styles.KEYWORD),
    /** The <code>super</code> keyword. */
    SUPER("'super'", Styles.KEYWORD),
    /** The <code>switch</code> keyword. */
    SWITCH("'switch'", Styles.KEYWORD),
    /** The <code>synchronized</code> keyword. */
    SYNCHRONIZED("'synchronized'", Styles.KEYWORD),
    /** The <code>this</code> keyword. */
    THIS("'this'", Styles.KEYWORD),
    /** The <code>throw</code> keyword. */
    THROW("'throw'", Styles.KEYWORD),
    /** The <code>throws</code> keyword. */
    THROWS("'throws'", Styles.KEYWORD),
    /** The <code>transient</code> keyword. */
    TRANSIENT("'transient'", Styles.KEYWORD),
    /** The <code>try</code> keyword. */
    TRY("'try'", Styles.KEYWORD),
    /** The <code>void</code> keyword. */
    VOID("'void'", Styles.KEYWORD),
    /** The <code>volatile</code> keyword. */
    VOLATILE("'volatile'", Styles.KEYWORD),
    /** The <code>while</code> keyword. */
    WHILE("'while'", Styles.KEYWORD),
    
    /** The <code>null</code> reserved word. */
    NULL("'null'", Styles.LITERAL),

    /** The '<tt>{</tt>' separator. */
    LBRACE("'{'", Styles.SEPARATOR),
    /** The '<tt>}</tt>' separator. */
    RBRACE("'}'", Styles.SEPARATOR),
    /** The '<tt>[</tt>' separator. */
    LBRACKET("'['", Styles.SEPARATOR),
    /** The '<tt>]</tt>' separator. */
    RBRACKET("']'", Styles.SEPARATOR),
    /** The '<tt>(</tt>' separator. */
    LPAREN("'('", Styles.SEPARATOR),
    /** The '<tt>)</tt>' separator. */
    RPAREN("')'", Styles.SEPARATOR),
    /** The '<tt>;</tt>' separator. */
    SEMICOLON("';'", Styles.SEPARATOR),
    /** The '<tt>,</tt>' separator. */
    COMMA("','", Styles.SEPARATOR),
    /** The '<tt>.</tt>' separator. */
    PERIOD("'.'", Styles.SEPARATOR),

    /** The '<tt>&lt;</tt>' operator. */
    LANGLE("'<'", Styles.OPERATOR),
    /** The '<tt>&gt;</tt>' operator. */
    RANGLE("'>'", Styles.OPERATOR),
    /** The '<tt>:</tt>' operator, in the ternary <tt>?:</tt> operation. */
    COLON("':'", Styles.SEPARATOR),
    /** The '<tt>=</tt>' operator. */
    ASSIGN("'='", Styles.OPERATOR),
    /** The '<tt>+</tt>' operator. */
    PLUS("'+'", Styles.OPERATOR),
    /** The '<tt>-</tt>' operator. */
    MINUS("'-'", Styles.OPERATOR),
    /** The '<tt>*</tt>' operator. */
    STAR("'*'", Styles.OPERATOR),
    /** The '<tt>/</tt>' operator. */
    SLASH("'/'", Styles.OPERATOR),
    /** The '<tt>%</tt>' operator. */
    PERCENT("'%'", Styles.OPERATOR),
    /** The '<tt>|</tt>' operator. */
    VERTICAL_BAR("'|'", Styles.OPERATOR),
    /** The '<tt>&amp;</tt>' operator. */
    AMPERSAND("'&'", Styles.OPERATOR),
    /** The '<tt>^</tt>' operator. */
    CARET("'^'", Styles.OPERATOR),
    /** The '<tt>~</tt>' operator. */
    TILDE("'~'", Styles.OPERATOR),
    /** The '<tt>!</tt>' operator. */
    EXCLAMATION("'!'", Styles.OPERATOR),
    /** The '<tt>?</tt>' operator, appearing in the <tt>?:</tt> operation. */
    QUESTION("'?'", Styles.OPERATOR),
    
    /** The '<tt>==</tt>' operator. */
    IS_EQUAL("'=='", Styles.OPERATOR),
    /** The '<tt>&lt;=</tt>' operator. */
    IS_LESS_EQUAL("'<='", Styles.OPERATOR),
    /** The '<tt>&gt;=</tt>' operator. */
    IS_GREATER_EQUAL("'>='", Styles.OPERATOR),
    /** The '<tt>!=</tt>' operator. */
    IS_NOT_EQUAL("'!='", Styles.OPERATOR),
    /** The '<tt>++</tt>' operator. */
    INCREMENT("'++'", Styles.OPERATOR),
    /** The '<tt>--</tt>' operator. */
    DECREMENT("'--'", Styles.OPERATOR),
    /** The '<tt>&lt;&lt;</tt>' operator. */
    SHIFT_LEFT("'<<'", Styles.OPERATOR),
    /** The '<tt>&gt;&gt;</tt>' operator. */
    SHIFT_RIGHT("'>>'", Styles.OPERATOR),
    /** The '<tt>&gt;&gt;&gt;</tt>' operator. */
    SHIFT_RIGHT_LOGICAL("'>>>'", Styles.OPERATOR),
    /** The '<tt>&amp;&amp;</tt>' operator. */
    LOGICAL_AND("'&&'", Styles.OPERATOR),
    /** The '<tt>||</tt>' operator. */
    LOGICAL_OR("'||'", Styles.OPERATOR),
    /** The '<tt>+=</tt>' operator. */
    ADD_ASSIGN("'+='", Styles.OPERATOR),
    /** The '<tt>-=</tt>' operator. */
    SUBTRACT_ASSIGN("'-='", Styles.OPERATOR),
    /** The '<tt>*=</tt>' operator. */
    MULTIPLY_ASSIGN("'*='", Styles.OPERATOR),
    /** The '<tt>/=</tt>' operator. */
    DIVIDE_ASSIGN("'/='", Styles.OPERATOR),
    /** The '<tt>%=</tt>' operator. */
    MOD_ASSIGN("'%='", Styles.OPERATOR),
    /** The '<tt>&amp;=</tt>' operator. */
    AND_ASSIGN("'&='", Styles.OPERATOR),
    /** The '<tt>|=</tt>' operator. */
    OR_ASSIGN("'|='", Styles.OPERATOR),
    /** The '<tt>^=</tt>' operator. */
    XOR_ASSIGN("'^='", Styles.OPERATOR),
    /** The '<tt>&lt;&lt;=</tt>' operator. */
    SHIFT_LEFT_ASSIGN("'<<='", Styles.OPERATOR),
    /** The '<tt>&gt;&gt;=</tt>' operator. */
    SHIFT_RIGHT_ASSIGN("'>>='", Styles.OPERATOR),
    /** The '<tt>&gt;&gt;&gt;=</tt>' operator. */
    SHIFT_RIGHT_LOGICAL_ASSIGN("'>>>='", Styles.OPERATOR),

    /** A <code>String</code> value. */
    STRING_LITERAL("String value", Styles.LITERAL),
    /** A <code>char</code> value. */
    CHARACTER_LITERAL("char value", Styles.LITERAL),
    /** A <code>boolean</code> value. */
    BOOLEAN_LITERAL("boolean value", Styles.LITERAL),
    /** An <code>int</code> value. */
    INTEGER_LITERAL("int value", Styles.LITERAL),
    /** A <code>long</code> value. */
    LONG_LITERAL("long value", Styles.LITERAL),
    /** A <code>float</code> value. */
    FLOAT_LITERAL("float value", Styles.LITERAL),
    /** A <code>double</code> value. */
    DOUBLE_LITERAL("double value", Styles.LITERAL),

    /** An identifier. */
    IDENTIFIER("identifier", Styles.IDENTIFIER),
    /** A Javadoc comment. */
    JAVADOC("'/** ... */'", Styles.COMMENT),
    /** A ``traditional'' comment beginning with "<tt>/*</tt>". */
    COMMENT_TRADITIONAL("'/* ... */'", Styles.COMMENT),
    /** A single-line comment beginning with "<tt>//</tt>". */
    COMMENT_END_OF_LINE("'// ...'", Styles.COMMENT),
    /** A sequence of invalid characters. */
    ERROR("ERROR", Styles.ERROR);

    /** Stores a map from keyword strings to their corresponding
     * <code>JavaTokenType</code>s.
     */
    private static HashMap<String, JavaTokenType> keywords;

    static {
        // initialize <code>keywords</code> by going through all
        // values; any beginning with a single-quotation and a
        // character are keywords that we will add into our map.
        keywords = new HashMap<String, JavaTokenType>();
        for(JavaTokenType t : values()) {
            String desc = t.getDescriptor();
            if(desc.length() > 2 && desc.charAt(0) == '\''
                    && Character.isLetter(desc.charAt(1))) {
                String keyword = desc.substring(1, desc.length() - 1);
                keywords.put(keyword, t);
            }
        }
        keywords.put("true", BOOLEAN_LITERAL);
        keywords.put("false", BOOLEAN_LITERAL);
    }

    /** A terse description of what sort of token this is. */
    private String descriptor;
    
    /** The style in which tokens of this type should be drawn. */
    private AttributeSet attrs;

    /**
     * Constructs a Java token type with the given descriptor and style.
     * 
     * @param descriptor  a terse description of the token type.
     * @param attrs  the style in which tokens of the type should be drawn.
     */
    private JavaTokenType(String descriptor, AttributeSet attrs) {
        this.descriptor = descriptor;
        this.attrs = attrs;
    }

    /**
     * Returns a terse description of what sort of token this
     * corresponds to.
     * 
     * @return the terse description.
     */
    public String getDescriptor() {
        return descriptor;
    }
    
    /**
     * Return the style in which tokens of this type should be drawn.
     * 
     * @return the style for tokens of this type.
     */
    public AttributeSet getAttributeSet() {
        return attrs;
    }

    /**
     * Returns the token type corresponding to this word-like string,
     * which should be either an identifier, a keyword, or reserved
     * word.
     * 
     * @param text  the word to look up.
     * @return the corresponding token type.
     */
    public static JavaTokenType forKeyword(String text) {
        JavaTokenType ret = keywords.get(text);
        return ret == null ? IDENTIFIER : ret;
    }
}
