/*
 * Copyright (c) 2005, Carl Burch.
 * 
 * This file is part of the com.cburch.editor package. The latest
 * version is available at http://www.cburch.com/proj/editor/.
 *
 * The com.cburch.editor package is free software; you can redistribute
 * it and/or modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * The com.cburch.editor package is distributed in the hope that it will
 * be useful, but WITHOUT ANY WARRANTY; without even the implied warranty
 * of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with the com.cburch.editor package; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301  USA
 */
 
 package com.cburch.editor.tokens;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.Iterator;
import java.util.List;
import java.util.ListIterator;

import javax.swing.text.Position;

import com.cburch.editor.util.Positions;

/**
 * Maintains the list of tokens. All modifying <tt>List</tt> methods
 * are prohibited to ensure that only a tokenizer will have access
 * to modify the list. These tokens will be stored in the order they
 * appear in the underlying document.
 *  
 * @author Carl Burch
 * @version 0.1 2005-05-31
 *
 * @param <T> the type of token maintained in the list.
 */
public class TokenList<T extends BasicToken> implements List<T> {
    /** A comparator for comparing the beginning offsets of tokens. */
    private static final Comparator<BasicToken> compareBeginOffsets = new Comparator<BasicToken>() {
        public int compare(BasicToken a, BasicToken b) {
            int ao = a.getBeginOffset();
            int bo = b.getBeginOffset();
            if(ao < bo) return -1;
            if(ao > bo) return 1;
            return 0;
        }
    };

    /** A comparator for comparing the ending offsets of tokens. */
    private static final Comparator<BasicToken> compareEndOffsets = new Comparator<BasicToken>() {
        public int compare(BasicToken a, BasicToken b) {
            int ao = a.getEndOffset();
            int bo = b.getEndOffset();
            if(ao < bo) return -1;
            if(ao > bo) return 1;
            return 0;
        }
    };

    /**
     * This class exists only so that we can get access to
     * <code>ArrayList</code>'s <code>removeRange</code> method.  
     */
    static class Data<V extends BasicToken> extends ArrayList<V> {
        protected void removeRange(int start, int end) {
            super.removeRange(start, end);
        }
    }
    
    /** Stores the list of all tokens. This has default protection
     * only so that <code>MutableTokenList</code> can get to it. */
    Data<T> data = new Data<T>();
    
    /** Holds an unmodifiable view of the above. This is important
     * to prevent modifications via the <code>iterator</code> method,
     * for example.  */
    private List<T> view = Collections.unmodifiableList(data);
    
    /** Constructs an empty list of tokens. */
    TokenList() { }
    
    /**
     * Constructs a list of tokens initially holding the given values.
     * 
     * @param initialValues  the tokens to have in the list initially.
     */
    TokenList(List<? extends T> initialValues) {
        if(initialValues != null) data.addAll(initialValues);
    }
    
    /**
     * Clears the list of all tokens. This has default protection
     * to ensure that classes outside the package cannot modify the
     * list.
     */
    void doClear() {
        data.clear();
    }
    
    /**
     * Replaces a range of the list with a different list instead.
     * 
     * @param from  the first index to be removed.
     * @param to    the first index to be retained, one after the
     *    index of the last index to be removed.
     * @param newTokens  the tokens to be inserted in place of the
     *    removed tokens.
     */
    void replace(int from, int to, List<T> newTokens) {
        if(from < 0 || to > size() || from > to) {
            throw new IllegalArgumentException("Cannot replace from " + from + " to " + to);
        }
        
        int newTo = from + (newTokens == null ? 0 : newTokens.size());
        if(newTo < to) {
            for(int i = from; i < newTo; i++) data.set(i, newTokens.get(i - from));
            data.removeRange(newTo, to);
        } else {
            for(int i = from; i < to; i++) data.set(i, newTokens.get(i - from));
            if(to < newTo) data.addAll(to, newTokens.subList(to - from, newTokens.size()));
        }
    }

    public int size() { return data.size(); }
    public boolean isEmpty() { return data.isEmpty(); }
    public boolean contains(Object arg0) { return data.contains(arg0); }
    public Object[] toArray() { return data.toArray(); }
    public <T> T[] toArray(T[] arg0) { return data.toArray(arg0); }
    public boolean containsAll(Collection<?> arg0) { return data.containsAll(arg0); }
    public T get(int arg0) { return data.get(arg0); }
    public int indexOf(Object arg0) { return data.indexOf(arg0); }
    public int lastIndexOf(Object arg0) { return data.lastIndexOf(arg0); }

    public Iterator<T> iterator() { return view.iterator(); }
    public ListIterator<T> listIterator() { return view.listIterator(); }
    public ListIterator<T> listIterator(int arg0) { return view.listIterator(arg0); }
    public List<T> subList(int arg0, int arg1) { return view.subList(arg0, arg1); }

    /** Unsupported operation. */ public boolean add(T arg0) { throw new UnsupportedOperationException("TokenList.add [1]"); }
    /** Unsupported operation. */ public void add(int arg0, T arg1) { throw new UnsupportedOperationException("TokenList.add [2]"); }
    /** Unsupported operation. */ public boolean addAll(Collection<? extends T> arg0) { throw new UnsupportedOperationException("TokenList.addAll [1]"); }
    /** Unsupported operation. */ public boolean addAll(int arg0, Collection<? extends T> arg1) { throw new UnsupportedOperationException("TokenList.addAll [2]"); }
    /** Unsupported operation. */ public boolean removeAll(Collection<?> arg0) { throw new UnsupportedOperationException("TokenList.removeAll"); }
    /** Unsupported operation. */ public boolean retainAll(Collection<?> arg0) { throw new UnsupportedOperationException("TokenList.retainAll"); }
    /** Unsupported operation. */ public void clear() { throw new UnsupportedOperationException("TokenList.clear"); }
    /** Unsupported operation. */ public T set(int arg0, T arg1) { throw new UnsupportedOperationException("TokenList.set"); }
    /** Unsupported operation. */ public T remove(int arg0) { throw new UnsupportedOperationException("TokenList.remove [int]"); }
    /** Unsupported operation. */ public boolean remove(Object arg0) { throw new UnsupportedOperationException("TokenList.remove [Object]"); }

    /**
     * Returns the index of the token that includes the given
     * offset, or -1 if no token includes that offset.
     * 
     * @param offset  the query offset.
     * @return  the index of the token including the offset,
     *     or -1 if there is no such token.
     */
    public int getIndexContaining(int offset) {
        return getIndexContaining(Positions.createDummy(offset));
    }

    /**
     * Returns the index of the token that includes the given
     * position, or -1 if no token includes that position.
     * 
     * @param pos  the query position.
     * @return  the index of the token including the position,
     *     or -1 if there is no such token.
     */
    public int getIndexContaining(Position pos) {
        // We'll search a range of tokens, just to make sure we
        // don't miss the desired token.
        int start = Collections.binarySearch(this,
                new BasicToken(pos, ""), compareEndOffsets);
        if(start < 0) start = -(start + 1);
        start = Math.max(0, start - 2);
        int stop = Math.min(size(), start + 5);
        
        // Now search for the requested offset.
        int offs = pos.getOffset();
        for(int i = start; i < stop; i++) {
            BasicToken t = data.get(i);
            if(offs >= t.getBeginOffset() && offs < t.getEndOffset()) {
                return i;
            }
        }
        return -1;
    }

    /**
     * Returns the index of the first token that begins at or after
     * the given offset. <strong>Note:</strong> This method is not
     * thoroughly tested.
     * 
     * @param offset  the minimum offset.
     * @return  the index of the first token beginning at or after
     *   the offset.
     */
    public int getIndexStartingAfter(int offset) {
        return getIndexStartingAfter(Positions.createDummy(offset));
    }

    /**
     * Returns the index of the first token that begins at or after
     * the given position. <strong>Note:</strong> This method is not
     * thoroughly tested.
     * 
     * @param pos  the minimum position.
     * @return  the index of the first token beginning at or after
     *   the position.
     */
    public int getIndexStartingAfter(Position pos) {
        int ret = Collections.binarySearch(data,
                new BasicToken(pos, ""), compareBeginOffsets);
        if(ret < 0) ret = -(ret + 1);
        return ret;
    }
    
    /**
     * Returns the index of the first token that ends at or before
     * the given offset. <strong>Note:</strong> This method is not
     * thoroughly tested.
     * 
     * @param offset  the maximum offset.
     * @return  the index of the first token ending at or before
     *   the offset.
     */
    public int getIndexEndingBefore(int offset) {
        return getIndexEndingBefore(Positions.createDummy(offset));
    }

    /**
     * Returns the index of the first token that ends at or before
     * the given position. <strong>Note:</strong> This method is not
     * thoroughly tested.
     * 
     * @param pos  the maximum position.
     * @return  the index of the first token ending at or before
     *   the position.
     */
    public int getIndexEndingBefore(Position pos) {
        int ret = Collections.binarySearch(data,
                new BasicToken(pos, ""), compareEndOffsets);
        if(ret < 0) ret = -(ret + 1);
        return ret;
    }

}
