/*
 * Copyright (c) 2005, Carl Burch.
 * 
 * This file is part of the com.cburch.editor package. The latest
 * version is available at http://www.cburch.com/proj/editor/.
 *
 * The com.cburch.editor package is free software; you can redistribute
 * it and/or modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * The com.cburch.editor package is distributed in the hope that it will
 * be useful, but WITHOUT ANY WARRANTY; without even the implied warranty
 * of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with the com.cburch.editor package; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301  USA
 */
 
 package com.cburch.editor.tokens;

import java.util.List;

/**
 * Describes information about a modification to the list of
 * tokens maintained by the tokenizer. As an example of what
 * the event describes, suppose that we had the following tokens.
 * <pre>
 *    a  b  c  d  e  f  g  h
 * </pre>
 * If the document were edited to remove <tt>d</tt> and
 * <tt>e</tt> the surrounding spaces, we would then have
 * <pre>
 *    a  b  cf  g  h
 * </pre>
 * The <tt>TokenizerEvent</tt> describing this modification would
 * report that there was a modification beginning at index 2 (the
 * index of the first token removed), the list of tokens
 * removed is {<tt>c</tt>,<tt>d</tt>,<tt>e</tt>,<tt>f</tt>},
 * and the list of tokens added is {<tt>cf</tt>}.</p>
 * 
 * <p>The tokenizer tries to be quite conservative in the size of the
 * changes reported; but for each document edit, it will never
 * produce more than one event.
 * 
 * @author Carl Burch
 * @version 0.1 2005-05-31
 *
 * @param <T> The type of token found in the tokenizer.
 */
public class TokenizerEvent<T extends BasicToken> {
    private Tokenizer<T> source;
    private int rangeBegin;
    private List<T> oldTokens;
    private List<T> newTokens;
    
    /**
     * Constructs a description of a modification to a tokenizer's
     * list of tokens. Note that the two lists may be of different
     * lengths, and quite likely neither will extend to the end of
     * the underlying list.
     * 
     * @param source  the tokenizer that initiated the event.
     * @param rangeBegin  the index within the token list where the
     *     replacement began.
     * @param oldTokens  the tokens that were previously in the location.
     * @param newTokens  the tokens that are now in the location.
     */
    public TokenizerEvent(Tokenizer<T> source, int rangeBegin,
            List<T> oldTokens, List<T> newTokens) {
        this.source = source;
        this.rangeBegin = rangeBegin;
        this.oldTokens = oldTokens;
        this.newTokens = newTokens;
    }

    /**
     * Returns the tokenizer that performed the modification.
     * 
     * @return the initiating tokenizer.
     */
    public Tokenizer<T> getSource() {
        return source;
    }
    
    /**
     * Returns the index of the token list where the modifications
     * begin.
     * 
     * @return the starting index for the modifications.
     */
    public int getFromIndex() {
        return rangeBegin;
    }
    
    /**
     * Returns a segment of the list of tokens, all of which were
     * removed from the list.
     * 
     * @return the segment of tokens removed.
     */
    public List<T> getOldTokens() {
        return oldTokens;
    }
    
    /**
     * Returns a segment of the list of tokens, all of which were
     * inserted to the list at the given location.
     * 
     * @return the segment of tokens inserted
     */
    public List<T> getNewTokens() {
        return newTokens;
    }
}
