/*
 * Copyright (c) 2005, Carl Burch.
 * 
 * This file is part of the com.cburch.editor package. The latest
 * version is available at http://www.cburch.com/proj/editor/.
 *
 * The com.cburch.editor package is free software; you can redistribute
 * it and/or modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * The com.cburch.editor package is distributed in the hope that it will
 * be useful, but WITHOUT ANY WARRANTY; without even the implied warranty
 * of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with the com.cburch.editor package; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301  USA
 */
 
 package com.cburch.editor.util;

import java.util.Comparator;

import javax.swing.text.Position;

/**
 * This class is a collection exclusively of class members that
 * are helpful in dealing with javax.swing.text's
 * <code>Position</code> class.
 * 
 * @author Carl Burch
 * @version 0.1 2005-05-31
 *
 */
public class Positions {
    /**
     * The constructor method is disabled here because the class
     * is meant to contain only class members.
     */
    private Positions() { }

    /**
     * Remembers the comparator, so that only one is ever created.
     */
    private static Comparator<Position> comparator = null;
    
    /**
     * Returns a comparator that compares two positions' offsets.
     * When two distinct <code>Position</code> objects are at the
     * same offset, the comparator will return a zero value. This
     * is technically erroneous, because two positions may be
     * different at one invocation and later, due to edits in the
     * document, become the same.
     * 
     * @return the comparator.
     */
    public static Comparator<Position> getComparator() {
        if(comparator == null) {
            comparator = new Comparator<Position>() {
                public int compare(Position a, Position b) {
                    int ao = a.getOffset();
                    int bo = b.getOffset();
                    if(ao < bo) return -1;
                    if(ao > bo) return 1;
                    return a.hashCode() - b.hashCode();
                }
            };
        }
        return comparator;
    }

    /**
     * Creates and returns a <code>Position</code> object that is
     * not associated with any <code>Document</code>. As a result,
     * the associated position does not change.
     * 
     * @param offset  the offset of the returned position.
     * @return a <code>Position</code> object wrapped around the
     *     given offset.
     */
    public static Position createDummy(final int offset) {
        return new Position() {
            int myOffset = offset;
            public int getOffset() { return myOffset; }
        };
    }
}
