/*
 * Copyright (c) 2005, Carl Burch.
 * 
 * This file is part of the com.cburch.editor package. The latest
 * version is available at http://www.cburch.com/proj/editor/.
 *
 * The com.cburch.editor package is free software; you can redistribute
 * it and/or modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * The com.cburch.editor package is distributed in the hope that it will
 * be useful, but WITHOUT ANY WARRANTY; without even the implied warranty
 * of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with the com.cburch.editor package; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301  USA
 */
 
 package com.cburch.editor;

import java.awt.Dimension;
import java.awt.Point;
import java.awt.event.InputEvent;
import java.awt.event.KeyEvent;
import java.awt.event.MouseEvent;

import javax.swing.InputMap;
import javax.swing.JTextPane;
import javax.swing.KeyStroke;
import javax.swing.text.DefaultEditorKit;
import javax.swing.text.DefaultStyledDocument;
import javax.swing.text.Document;

import com.cburch.editor.tokens.TokenList;
import com.cburch.editor.tokens.Tokenizer;

/**
 * Implements a text editor for which various features
 * common to high-quality text editors can be installed.
 * 
 * @author Carl Burch
 * @version 0.1 2005-05-31
 */
public class Editor extends JTextPane {
    private Scanner<? extends Token> scanner = null;
    private Tokenizer<Token> tokenizer;
    private boolean errorToolTipsEnabled = false;

    /**
     * Constructs an empty text editor.
     */
    public Editor() {
        // We need a StyledDocument so that the syntax
        // highlighting can work.
        setDocument(new DefaultStyledDocument());

        // By default, Java has the home and end keys go to the
        // beginning and end of the document, rather than
        // the beginning and end of the current line. This
        // reconfigures the keys to work more appropriately.
        InputMap inputs = getInputMap();
        inputs.put(KeyStroke.getKeyStroke(KeyEvent.VK_HOME, 0),
                DefaultEditorKit.beginLineAction);
        inputs.put(KeyStroke.getKeyStroke(KeyEvent.VK_HOME, InputEvent.SHIFT_MASK),
                DefaultEditorKit.selectionBeginLineAction);
        inputs.put(KeyStroke.getKeyStroke(KeyEvent.VK_END, 0),
                DefaultEditorKit.endLineAction);
        inputs.put(KeyStroke.getKeyStroke(KeyEvent.VK_END, InputEvent.SHIFT_MASK),
                DefaultEditorKit.selectionEndLineAction);
    }
    
    /**
     * Returns the scanner currently associated with this
     * editor.
     * 
     * @return the editor's current scanner.
     */
    public Scanner getScanner() {
        return scanner;
    }
    
    /**
     * Changes the scanner used for breaking the editor into
     * tokens.
     * 
     * @param value  the scanner to use.
     */
    public void setScanner(Scanner<? extends Token> value) {
        scanner = value;
        if(tokenizer != null) tokenizer.setScanner(value);
    }
    
    /**
     * Returns <code>true</code> if the editor is configured to
     * display tool tips when the mouse hovers over a token with an
     * associated error message.
     * 
     * @return <code>true</code> if it will display error messages,
     *     <code>false</code> if it will not.
     */
    public boolean isErrorToolTipsEnabled() {
        return errorToolTipsEnabled;
    }
    
    /**
     * Configures whether the editor should display tool tips
     * when the mouse hovers over a token with an associated error
     * message.
     * @param value
     */
    public void setErrorToolTipsEnabled(boolean value) {
        errorToolTipsEnabled = value;
        
        // No tool tips are displayed unless the overall
        // component has a tool tip set. We'll just use an empty
        // string.
        String text = getToolTipText();
        if(errorToolTipsEnabled) {
            if(text == null) setToolTipText("");
        } else {
            if(text != null && text.equals("")) setToolTipText(null);
        }
    }
    
    /**
     * Returns the tokenizer used to track the tokens that
     * exist in the document.
     * 
     * @return the tokenizer holding the tokens.
     */
    public Tokenizer<? extends Token> getTokenizer() {
        if(tokenizer == null) {
            tokenizer = new Tokenizer<Token>(getDocument(), scanner);
        }
        return tokenizer;
    }
    
    /**
     * Overrides <code>JTextPane</code>'s <code>setDocument</code>
     * method so that we can adapt the tokenizer to work with the
     * newly current document instead.
     * 
     * @param value  the document to display in this text pane.
     */
    public void setDocument(Document value) {
        super.setDocument(value);
        if(tokenizer != null) tokenizer.setDocument(getDocument());
    }
    
    /**
     * Overrides <code>JTextPane</code>'s method so that we can
     * disable word-wrapping.
     */
    public boolean getScrollableTracksViewportWidth() {
        return false;
    }
    
    /**
     * Overrides <code>JTextPane</code>'s method so that we can
     * disable word-wrapping.
     */
    public void setSize(Dimension d) {
        if(d.width < WIDTH) d.width = WIDTH;
        super.setSize(d);
    }
    
    /**
     * Overrides <code>JComponent</code>'s <code>getToolTipText</code>
     * method so that we can potentially display error messages.
     * 
     * @param event  a mouse event giving the coordinates where we
     *     should search for a token.
     * @return the string to display as the tool tip.
     */
    public String getToolTipText(MouseEvent event) {
        if(!errorToolTipsEnabled) return null;
        int offs = viewToModel(new Point(event.getX(), event.getY()));
        TokenList<? extends Token<?>> tokens = getTokenizer().getTokenList();
        int i = tokens.getIndexContaining(offs);
        if(i < 0) return null;
        String ret = tokens.get(i).getErrorMessage();
        if(ret == null) return null;
        return ret;
    }
}
