/*
 * Copyright (c) 2005, Carl Burch.
 * 
 * This file is part of the com.cburch.editor package. The latest
 * version is available at http://www.cburch.com/proj/editor/.
 *
 * The com.cburch.editor package is free software; you can redistribute
 * it and/or modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * The com.cburch.editor package is distributed in the hope that it will
 * be useful, but WITHOUT ANY WARRANTY; without even the implied warranty
 * of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with the com.cburch.editor package; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301  USA
 */
 
 package com.cburch.editor;

import java.awt.BorderLayout;
import java.awt.Dimension;

import javax.swing.JPanel;
import javax.swing.JScrollPane;

/**
 * A component that places an editor within a scroll pane,
 * with line numbers installed.
 * 
 * @author Carl Burch
 * @version 0.1 2005-05-31
 *
 */
public class EditorPanel extends JPanel {
    /** The scroll pane within this panel. */
    private JScrollPane scrollPane;
    
    /** The editor within this panel. */
    private Editor editor;
    
    /** The line numbers within this panel. */
    private LineNumbers lineNumbers;
    
    /**
     * Constructs a panel containing an uninitialized editor.
     */
    public EditorPanel() {
        this(new Editor());
    }

    /**
     * Constructs a panel containing the given editor.
     * 
     * @param editor  the editor to include within this panel.
     */
    public EditorPanel(Editor editor) {
        super(new BorderLayout());
        this.editor = editor;
        this.lineNumbers = new LineNumbers(editor);
        this.scrollPane = new JScrollPane(editor);
        
        scrollPane.setRowHeader(lineNumbers);
        scrollPane.setPreferredSize(new Dimension(620, 460));
        scrollPane.getViewport().setBackground(editor.getBackground());
        add(scrollPane);
    }
    
    /**
     * Returns the editor currently within this panel.
     * 
     * @return this panel's editor.
     */
    public Editor getEditor() {
        return editor;
    }
    
    /**
     * Changes the editor currently within this panel.
     * 
     * @param value  the editor to place into this panel.
     */
    public void setEditor(Editor value) {
        editor = value;
        lineNumbers.setJTextComponent(value);
        scrollPane.setViewportView(value);
    }
    
    /**
     * Returns the scroll pane wrapped around the editor.
     * 
     * @return the scroll pane.
     */
    public JScrollPane getJScrollPane() {
        return scrollPane;
    }
    
    /**
     * Returns the line-numbering object associated with
     * the editor.
     * 
     * @return the associated line-numbering object.
     */
    public LineNumbers getLineNumbers() {
        return lineNumbers;
    }
    
    /**
     * Returns <code>true</code> if the line numbers are
     * currently being shown.
     * 
     * @return <code>true</code> if line numbers are visible. 
     */
    public boolean isLineNumbersVisible() {
        return scrollPane.getRowHeader() == lineNumbers;
    }
    
    /**
     * Configures whether the line numbers should be displayed.
     * 
     * @param value  <code>true</code> if line numbers should be
     *    visible.
     */
    public void setLineNumbersVisible(boolean value) {
        scrollPane.setRowHeader(value ? lineNumbers : null);
    }
}
