/*
 * Copyright (c) 2005, Carl Burch.
 * 
 * This file is part of the com.cburch.editor package. The latest
 * version is available at http://www.cburch.com/proj/editor/.
 *
 * The com.cburch.editor package is free software; you can redistribute
 * it and/or modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * The com.cburch.editor package is distributed in the hope that it will
 * be useful, but WITHOUT ANY WARRANTY; without even the implied warranty
 * of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with the com.cburch.editor package; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301  USA
 */
 
 package com.cburch.editor;

import java.awt.Color;
import java.awt.Dimension;
import java.awt.FontMetrics;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.Point;
import java.awt.Rectangle;
import java.awt.font.LineMetrics;
import java.util.Iterator;
import java.util.SortedSet;
import java.util.TreeSet;

import javax.swing.JViewport;
import javax.swing.event.DocumentEvent;
import javax.swing.event.DocumentListener;
import javax.swing.text.BadLocationException;
import javax.swing.text.Document;
import javax.swing.text.JTextComponent;
import javax.swing.text.Position;

import com.cburch.editor.util.Positions;

/**
 * Tracks the lines within a document, and enables them to be
 * displayed along with the document. Even when line numbers
 * aren't displayed, objects of this class are still useful for
 * converting from line numbers to offsets and vice versa.</p>
 * 
 * <p>Line numbers are by default numbered starting with 1, although
 * this can be configured via the <code>setFirstLine</code> method.
 * The <code>toOffset</code> and <code>fromOffset</code> methods
 * take this into account; the line numbers they work with are
 * not zero-indexed unless <code>setFirstLine</code> has been
 * configured thus.</p>
 * 
 * <p>This implementation has a small known bug: It does not identify
 * when the text component's document changes. If this should
 * happen, it will maintain the line numbers of the previous
 * document. You can get it to reset to the proper document using
 * its <code>setJTextComponent</code> method.
 * 
 * @author Carl Burch
 * @version 0.1 2005-05-31
 */
public class LineNumbers extends JViewport {
    /** Listens for changes to the document. */
    private class MyListener implements DocumentListener {
        /** Searches for new line breaks. */
        public void insertUpdate(DocumentEvent event) {
            int pos = event.getOffset();
            int len = event.getLength();
            int oldSize = lineBreaks.size();
            addAll(pos, len);
            if(lineBreaks.size() != oldSize) repaint();
        }

        /** Searches for removed line breaks. */
        public void removeUpdate(DocumentEvent event) {
            Document doc = event.getDocument();
            int offs = event.getOffset();
            int len = event.getLength();
            
            String text;
            try {
                text = doc.getText(offs, 1); 
            } catch(BadLocationException e) { text = " "; }
            boolean exists = (text.charAt(0) == '\n');
            
            SortedSet<Position> toRemove = lineBreaks.subSet(
                    Positions.createDummy(offs - 1),
                    Positions.createDummy(offs + 1));
            boolean found = false;
            boolean changed = false;
            for(Iterator<Position> it = toRemove.iterator(); it.hasNext(); ) {
                Position pos = it.next();
                if(pos.getOffset() == offs) {
                    if(!exists || found) {
                        changed = true;
                        it.remove();
                    } else {
                        found = true;
                    }
                }
            }
            if(changed) repaint();
        }

        /** Does nothing - this appears to be called only when
         * attributes change, which has nothing to do with line
         * breaks.
         */
        public void changedUpdate(DocumentEvent event) { }
    }
    
    /** The text pane whose line numbers we are tracking. */
    private JTextComponent textPane;
    
    /** The document to which we are listening. This will be the
     * same as <code>textPane.getDocument()</code> unless the
     * document associated with <code>textPane</code> changes without
     * this object being notified. */
    private Document document = null;
    
    /** The listener to the document. */
    private MyListener myListener = new MyListener();
    
    /** The set of locations where line breaks currently exist
     * within the document. */
    private TreeSet<Position> lineBreaks
        = new TreeSet<Position>(Positions.getComparator());

    /** The number to be associated with the first line of the document. */
    private int firstLine = 1;
    
    /** The distance that has been scrolled off the JTextComponent's top. */
    private int ytop = 0;
    
    /**
     * Constructs a line-numbering object for the given text component.
     * 
     * @param textPane  the text component whose line numbers we
     *    should track. 
     */
    public LineNumbers(JTextComponent textPane) {
        setJTextComponent(textPane);
        setBackground(textPane.getBackground());
        setPreferredSize(new Dimension(25, 25));
        setForeground(Color.GRAY);
        setFont(getFont().deriveFont(9.0f));
    }
    
    /**
     * Configures which text component's line numbers we should track.
     * 
     * @param value  the text component whom we should track.
     */
    public void setJTextComponent(JTextComponent value) {
        if(document != null) {
            document.removeDocumentListener(myListener);
            lineBreaks.clear();
        }
        textPane = value;
        document = textPane.getDocument();
        if(document != null) {
            addAll(0, document.getLength());
            document.addDocumentListener(myListener);
        }
    }
    
    /**
     * Returns the number currently associated with the first
     * line of the document.
     * 
     * @return the first line's number.
     */
    public int getFirstLine() { return firstLine; }
    
    /**
     * Changes the number associated with the document's first
     * line. This is 1 by default.
     * 
     * @param value  the first line's number.
     */
    public void setFirstLine(int value) { firstLine = value; }
    
    /**
     * The line number where the given offset is located within the
     * document. Note that line numbers are indexed starting at 1
     * by default.
     * 
     * @param offset  the offset whose line number is desired.
     * @return the number of the corresponding line, offset
     *    as indicated with <code>setFirstLine</code>.
     */
    public int fromOffset(int offset) {
        SortedSet<Position> headSet = lineBreaks.headSet(Positions.createDummy(offset));
        return firstLine + headSet.size();
    }
    
    /**
     * The offset of the first character in the given line. Note
     * that lines are not zero-indexed by default.
     * 
     * @param lineNumber  the line index in question, offset as
     *   configured with <code>setFirstLine</code>.
     * @return the offset of the first character within the line.
     */
    public int toOffset(int lineNumber) {
        Iterator<Position> it = lineBreaks.iterator();
        Position ret = null;
        for(int k = lineNumber - firstLine; k > 0 && it.hasNext(); k--) {
            ret = it.next();
        }
        return ret == null ? 0 : ret.getOffset() + 1;
    }
    
    /**
     * Overrides <code>JViewport</code>'s <code>setViewPosition</code>
     * method so that we can track the top of the viewport.
     * 
     * @param p  the point that is now the top of the viewport.
     */
    public void setViewPosition(Point p) {
        super.setViewPosition(p);
        if(p.y != ytop) {
            ytop = p.y;
            repaint();
        }
    }
    
    /**
     * Overrides <code>JViewport</code>'s <code>paintComponent</code>
     * method so that we can draw the line numbers appropriately.
     * 
     * @param g  the object we should use for painting line numbers.
     */
    public void paintComponent(Graphics g) {
        super.paintComponent(g);
        // TODO draw only what is in clipping rectangle
        Dimension sz = getSize();
        FontMetrics fm = g.getFontMetrics();
        LineMetrics pfm = textPane.getFont().getLineMetrics("0",
                ((Graphics2D) g).getFontRenderContext());
        
        int x = (int) sz.getWidth() - 2;
        int dy = (int) pfm.getAscent();
        
        int lineNumber = firstLine;
        try {
            String label = "" + lineNumber;
            Rectangle rect = textPane.modelToView(0);
            g.drawString(label, x - fm.stringWidth(label),
                    rect.y + dy - ytop);
            lineNumber++;
            for(Position pos : lineBreaks) {
                rect = textPane.modelToView(pos.getOffset() + 1);
            
                label = "" + lineNumber;
                g.drawString(label, x - fm.stringWidth(label),
                        rect.y + dy - ytop);
                lineNumber++;
            }
        } catch(BadLocationException e) { }
    }

    /**
     * Inserts all line breaks within the given range into our
     * set of line breaks.
     * 
     * @param pos  the offset where to begin searching.
     * @param len  the length of the segment to search.
     */
    private void addAll(int pos, int len) {
        try {
            // I'm shying away from reading the entire document
            // into a single string, as that seems rather inefficient.
            // Instead I go through the document, 64 characters at
            // a time.
            for(int offs = 0; offs < len; offs += 64) {
                String text = document.getText(pos + offs,
                        Math.min(len - offs, 64));
                for(int i = 0, n = text.length(); i < n; i++) {
                    if(text.charAt(i) == '\n') {
                        lineBreaks.add(document.createPosition(pos + offs + i));
                    }
                }
            }
        } catch(BadLocationException e) { }
    }
}
