/*
 * Copyright (c) 2005, Carl Burch.
 * 
 * This file is part of the com.cburch.editor package. The latest
 * version is available at http://www.cburch.com/proj/editor/.
 *
 * The com.cburch.editor package is free software; you can redistribute
 * it and/or modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * The com.cburch.editor package is distributed in the hope that it will
 * be useful, but WITHOUT ANY WARRANTY; without even the implied warranty
 * of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with the com.cburch.editor package; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301  USA
 */
 
 package com.cburch.editor;

import java.util.List;

import javax.swing.text.SimpleAttributeSet;
import javax.swing.text.StyledDocument;

import com.cburch.editor.tokens.Tokenizer;
import com.cburch.editor.tokens.TokenizerEvent;
import com.cburch.editor.tokens.TokenizerListener;

/**
 * Handles coloring tokens according to their corresponding styles.
 * For this to work, a scanner must be associated with the editor.
 * 
 * @author Carl Burch
 * @version 0.1 2005-05-31
 *
 */
public class SyntaxHighlighter {
    /** The attribute set used for removing highlights. */
    private static final SimpleAttributeSet nullAttributes = new SimpleAttributeSet();
    
    /** A listener for changes in the tokens of the document. */
    private class MyListener implements TokenizerListener<Token<?>> {
        public void rangeReplaced(TokenizerEvent<Token<?>> event) {
            StyledDocument doc = (StyledDocument) tokenizer.getDocument();
            
            // first we uncolor all the characters in the range of
            // removed tokens.
            List<Token<?>> oldTokens = event.getOldTokens();
            if(oldTokens.size() > 0) {
                // It may seem like we should start at the first
                // token's start offset. However, if the token has
                // been deleted, then the start offset is no longer
                // valid, as its value is based on the end offset.
                // We therefore start from the end offset, which
                // isn't correct but works well when the scanner
                // properly classifies all non-whitespace characters
                // into tokens.
                int start = oldTokens.get(0).getEndOffset();
                int stop = oldTokens.get(oldTokens.size() - 1).getEndOffset();
                doc.setCharacterAttributes(start, stop - start, nullAttributes, true);
            }
            
            // now color all the tokens that go in their place.
            colorAll(event.getNewTokens());
        }   
    }
    
    /** The tokenizer whose tokens we are coloring. */
    private Tokenizer<? extends Token> tokenizer;
    
    /** The listener we register with the tokenizer. */
    private MyListener myListener = new MyListener();
    
    /** Tracks whether the highlighting is currently enabled. */
    private boolean enabled = false;
    
    /**
     * Constructs a SyntaxHighlighter for the given editor.
     * 
     * @param editor  the editor whose contents we are to highlight.
     */
    public SyntaxHighlighter(Editor editor) {
        this.tokenizer = editor.getTokenizer();
        setEnabled(true);
    }
    
    /**
     * Returns <code>true</code> if syntax highlighting is enabled.
     * 
     * @return <code>true</code> if this highlighter is enabled,
     *    <code>false</code> otherwise.
     */
    public boolean isEnabled() {
        return enabled;
    }
    
    /**
     * Changes the enable/disable status of this syntax highlighter.
     * Disabling will remove all highlights; enabling will add all
     * highlights back.
     * 
     * @param value  <code>true</code> to enable the highlighter,
     *     <code>false</code> to disable.
     */
    public void setEnabled(boolean value) {
        if(enabled == value) return;
        enabled = value;
        if(enabled) {
            tokenizer.addTokenizerListener(myListener, false);
            colorAll(tokenizer.getTokenList());
        } else {
            tokenizer.removeTokenizerListener(myListener);
            StyledDocument doc = (StyledDocument) tokenizer.getDocument();
            doc.setCharacterAttributes(0, doc.getLength(), nullAttributes, true);
        }
    }

    /**
     * Colors all tokens in the given list according to the tokens'
     * respective styles.
     * 
     * @param tokens  The tokens we are to color.
     */
    private void colorAll(List<? extends Token<?>> tokens) {
        StyledDocument doc = (StyledDocument) tokenizer.getDocument();
        for(Token<?> t : tokens) {
            doc.setCharacterAttributes(t.getBeginOffset(),
                    t.getLength(), t.getAttributeSet(), true);
        }
    }
}
