/*
 * Copyright (c) 2005, Carl Burch.
 * 
 * This file is part of the com.cburch.editor package. The latest
 * version is available at http://www.cburch.com/proj/editor/.
 *
 * The com.cburch.editor package is free software; you can redistribute
 * it and/or modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * The com.cburch.editor package is distributed in the hope that it will
 * be useful, but WITHOUT ANY WARRANTY; without even the implied warranty
 * of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with the com.cburch.editor package; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301  USA
 */
 
 package com.cburch.editor;

import javax.swing.text.AttributeSet;
import javax.swing.text.Position;

import com.cburch.editor.tokens.BasicToken;

/**
 * Represents a token with an associated category.
 * 
 * @author Carl Burch
 * @version 0.1 2005-05-31
 *
 * @param <T> The enumeration associated as each token's category.
 */
public class Token<T extends TokenType> extends BasicToken {
    /** The category of this token. */
    private T type;
    
    /** The data associated with this token. See <code>getData</code>. */
    private Object data;
    
    /** A string describing an error associated with this token,
     * or null if there are no errors. */
    private String errorMessage;

    /**
     * Constructs a token with the given information. The associated
     * data value for this token will be <code>null</code>.
     * 
     * @param type  the category for this token.
     * @param endPosition  the position within the source document just past
     *     this token's last character.
     * @param text  the underlying text that this token represents.
     */
    public Token(T type, Position endPosition, String text) {
        this(type, endPosition, text, null);
    }

    /**
     * Constructs a token with the given information.
     * 
     * @param type  the category for this token.
     * @param endPosition  the position within the source document just past
     *     this token's last character.
     * @param text  the underlying text that this token represents.
     * @param data  the data value associated with this token.
     */
    public Token(T type, Position endPosition, String text,
            Object data) {
        super(endPosition, text);
        this.type = type;
        this.data = data;
    }

    /**
     * Returns the category associated with this token.
     * 
     * @return this token's category.
     */
    public T getType() {
        return type;
    }
    
    /**
     * Returns the style in which this token should be drawn. As
     * implemented here, it is simply whatever the token's type
     * says.
     * 
     * @return this token's style.
     */
    public AttributeSet getAttributeSet() {
        return type.getAttributeSet();
    }
    
    /**
     * Retrieves the data associated with this token. The data
     * value is mutable using the <code>setData</code> method,
     * and it can be used for many purposes. For literals, it might
     * be an object representing the literal values. For error
     * tokens, it might be a description of the error.</p>
     * 
     * <p>Note that BracketMatcher uses the data to track brackets'
     * corresponding tokens.
     * 
     * @return the data associated with this token.
     */
    public Object getData() {
        return data;
    }

    /**
     * Sets the data value associated with this token.
     * 
     * @param value  the value to associate with this token.
     */
    public void setData(Object value) {
        data = value;
    }
        
    /**
     * Retrieves the error mesage associated with this token. The message
     * is mutable using the <code>setErrorMessage</code> method.</p>
     * 
     * <p>Note that BracketMatcher sets the error message for
     * unmatched brackets.
     * 
     * @return the error message associated with this token.
     */
    public String getErrorMessage() {
        return errorMessage;
    }

    /**
     * Sets the error message associated with this token.
     * 
     * @param value  the error message to associate with this token.
     */
    public void setErrorMessage(String value) {
        errorMessage = value;
    }
        
    public String toString() {
        return type + "[" + getAbbreviatedText(20) + "]";
    }
    
    public boolean equals(Object other) {
        if(!(other instanceof Token)) return false;
        Token o = (Token) other;
        return super.equals(other)
            && this.type.equals(o.type);
    }

}
