/*
 * Copyright (c) 2005, Carl Burch.
 * 
 * This file is part of the com.cburch.editor package. The latest
 * version is available at http://www.cburch.com/proj/editor/.
 *
 * The com.cburch.editor package is free software; you can redistribute
 * it and/or modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * The com.cburch.editor package is distributed in the hope that it will
 * be useful, but WITHOUT ANY WARRANTY; without even the implied warranty
 * of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with the com.cburch.editor package; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301  USA
 */
 
 package com.cburch.editor.demo;

import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;

import javax.swing.ButtonGroup;
import javax.swing.JCheckBoxMenuItem;
import javax.swing.JMenu;
import javax.swing.JRadioButtonMenuItem;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;

import com.cburch.editor.AutoIndent;
import com.cburch.editor.BracketMatcher;
import com.cburch.editor.Editor;
import com.cburch.editor.EditorPanel;
import com.cburch.editor.SyntaxHighlighter;
import com.cburch.editor.scanners.HtmlScanner;
import com.cburch.editor.scanners.JavaScanner;
import com.cburch.editor.scanners.JavaTokenType;

/**
 * Implements the Options menu.
 * 
 * @author Carl Burch
 * @version 0.1 2005-05-31
 */
class OptionsMenu extends JMenu {
    /** Listens for selections of menu items. */
    private class MyListener implements ActionListener {
        public void actionPerformed(ActionEvent event) {
            Object source = event.getSource();
            if(source == lineNumbers) {
                editorPanel.setLineNumbersVisible(lineNumbers.isSelected());
            }
        }
    }
    
    /** Implements the code for enabling and disabling modes. */
    private abstract class ModeItem extends JRadioButtonMenuItem
            implements ChangeListener {
        /** The last state we observed. */ 
        private boolean lastState;
        
        /** Constructs the menu item with the given text. */
        ModeItem(String title) {
            super(title);
            addChangeListener(this);
            setSelected(false);
        }
        
        /** Forwards the changes to the <code>setEnabled</code> method. */
        public void stateChanged(ChangeEvent e) {
            if(isSelected() != lastState) {
                lastState = isSelected();
                setEnabled(lastState);
            }
        }
        
        /** Enables or disables the mode. */
        public abstract void setEnabled(boolean value);
    }

    /** The menu item for viewing plain text. */
    private class PlainItem extends ModeItem {
        /** Constructs the menu item. */
        PlainItem() {
            super("Plain");
            setEnabled(false);
        }
        
        /** Enables or disables the mode. */
        public void setEnabled(boolean value) {
            if(value) editorPanel.getEditor().setScanner(null);
            setSelected(value);
        }
    }
    
    /** The menu item for viewing Java text. */
    private class JavaItem extends ModeItem {
        private SyntaxHighlighter highlighter;
        private BracketMatcher braceMatcher;
        private BracketMatcher parenMatcher;
        private BracketMatcher brackMatcher;
        private AutoIndent autoindent;
        
        /** Constructs the menu item. */
        public JavaItem() {
            super("Java");
            Editor edit = editorPanel.getEditor(); 
            highlighter = new SyntaxHighlighter(edit);
            braceMatcher = new BracketMatcher(edit,
                    JavaTokenType.LBRACE, JavaTokenType.RBRACE,
                    "Corresponding brace is missing.");
            parenMatcher = new BracketMatcher(edit,
                    JavaTokenType.LPAREN, JavaTokenType.RPAREN,
                    "Corresponding parenthesis is missing.");
            brackMatcher = new BracketMatcher(edit,
                    JavaTokenType.LBRACKET, JavaTokenType.RBRACKET,
                    "Corresponding bracket is missing.");
            autoindent = new AutoIndent(edit,
                    JavaTokenType.LBRACE, JavaTokenType.RBRACE, '}');
            setEnabled(false);
        }
        
        /** Enables or disables the mode. */
        public void setEnabled(boolean value) {
            if(value) editorPanel.getEditor().setScanner(new JavaScanner());
            highlighter.setEnabled(value);
            braceMatcher.setCaretMatchHighlighted(value);
            braceMatcher.setUnmatchedHighlighted(value);
            parenMatcher.setCaretMatchHighlighted(value);
            parenMatcher.setUnmatchedHighlighted(value);
            brackMatcher.setCaretMatchHighlighted(value);
            brackMatcher.setUnmatchedHighlighted(value);
            autoindent.setEnabled(value);
            setSelected(value);
        }
    }
    
    /** The menu item for viewing Java text. */
    private class HtmlItem extends ModeItem {
        private SyntaxHighlighter highlighter;
        
        /** Constructs the menu item. */
        public HtmlItem() {
            super("HTML");
            Editor edit = editorPanel.getEditor(); 
            highlighter = new SyntaxHighlighter(edit);
            setEnabled(false);
        }
        
        /** Enables or disables the mode. */
        public void setEnabled(boolean value) {
            if(value) editorPanel.getEditor().setScanner(new HtmlScanner());
            highlighter.setEnabled(value);
            setSelected(value);
        }
    }
    
    private EditorPanel editorPanel;
    private MyListener myListener = new MyListener();
    private JCheckBoxMenuItem lineNumbers = new JCheckBoxMenuItem("Number Lines");
    private ModeItem[] modeItems;

    /**
     * Constructs the Options menu.
     * 
     * @param editorPanel  the editor controlled by this menu.
     */
    public OptionsMenu(EditorPanel editorPanel) {
        super("Options");
        this.editorPanel = editorPanel;

        JMenu text = new JMenu("Text Type");
        ButtonGroup bgroup = new ButtonGroup();
        modeItems = new ModeItem[] {
            new PlainItem(),
            new JavaItem(),
            new HtmlItem(),
        };
        for(int i = 0; i < modeItems.length; i++) {
            text.add(modeItems[i]);
            bgroup.add(modeItems[i]);
        }
        modeItems[1].setEnabled(true);
        
        add(lineNumbers);   lineNumbers.addActionListener(myListener);
        add(text);
        
        lineNumbers.setSelected(editorPanel.isLineNumbersVisible());
    }
}
