/*
 * Copyright (c) 2005, Carl Burch.
 * 
 * This file is part of the com.cburch.editor package. The latest
 * version is available at http://www.cburch.com/proj/editor/.
 *
 * The com.cburch.editor package is free software; you can redistribute
 * it and/or modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * The com.cburch.editor package is distributed in the hope that it will
 * be useful, but WITHOUT ANY WARRANTY; without even the implied warranty
 * of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with the com.cburch.editor package; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301  USA
 */
 
 package com.cburch.editor.scanners;

import java.awt.Color;

import javax.swing.text.AttributeSet;
import javax.swing.text.SimpleAttributeSet;
import javax.swing.text.StyleConstants;

/**
 * Defines convenience methods for defining styles by token types.
 * 
 * @author Carl Burch
 * @version 0.1 2005-05-31
 *
 */
class Styles {
    /**
     * The constructor method is disabled here because the class
     * is meant to contain only class members.
     */
    private Styles() { }

    /** A mask used for representing bold fonts. */
    public static final int BOLD = 1;
    /** A mask used for representing italic fonts. */
    public static final int ITALIC = 2;
    /** A mask used for representing underlined fonts. */
    public static final int UNDERLINE = 4;
    
    /** A maroon color. */
    public static final Color MAROON = new Color(0xB03060);
    /** A deep blue color. */
    public static final Color DARK_BLUE = new Color(0x000080);
    /** A dark green color. */
    public static final Color DARK_GREEN = Color.GREEN.darker();
    /** A dark purple color. */
    public static final Color DARK_PURPLE = new Color(0xA020F0).darker();
    
    //
    // General styles
    //  
    /** A style suitable for representing scanning errors in a document. */
    public static final AttributeSet ERROR = create(Color.RED, BOLD);
    
    /** A style suitable for valid sequences of unknown meaning in a document. */
    public static final AttributeSet UNKNOWN = create(Color.ORANGE);
    
    //
    // Styles having to do with HTML text.
    //
    public static final AttributeSet TAG = create(Color.BLUE, BOLD);
    public static final AttributeSet BAD_TAG = create(DARK_PURPLE, BOLD);
    public static final AttributeSet TEXT = create(Color.BLACK);
    public static final AttributeSet ANCHOR_TEXT = create(Color.BLACK, UNDERLINE);
    
    /*
    addStyle("body", Color.WHITE, Color.BLACK, false, false);
    addStyle("tag", Color.WHITE, Color.BLUE, true, false);
    addStyle("endtag", Color.WHITE, Color.BLUE, false, false);
    addStyle("reference", Color.WHITE, Color.BLACK, false, false);
    addStyle("name", Color.WHITE, maroon, true, false);
    addStyle("value", Color.WHITE, maroon, false, true);
    addStyle("text", Color.WHITE, Color.BLACK, true, false);
    addStyle("reservedWord", Color.WHITE, Color.BLUE, false, false);
    addStyle("identifier", Color.WHITE, Color.BLACK, false, false);
    addStyle("literal", Color.WHITE, maroon, false, false);
    addStyle("separator", Color.WHITE, darkBlue, false, false);
    addStyle("operator", Color.WHITE, Color.BLACK, true, false);
    addStyle("comment", Color.WHITE, darkGreen, false, false);
    addStyle("preprocessor", Color.WHITE, darkPurple, false, false);
    addStyle("whitespace", Color.WHITE, Color.BLACK, false, false);
    addStyle("error", Color.WHITE, Color.RED, false, false);
    addStyle("unknown", Color.WHITE, Color.ORANGE, false, false);
    addStyle("grayedOut", Color.WHITE, Color.GRAY, false, false);
    */

    //
    // Styles having to do with program text.
    //
    /** A style suitable for representing basic text in a document. */
    public static final AttributeSet BODY = create(Color.BLACK);
    
    /** A style suitable for representing keywords in a program. */
    public static final AttributeSet KEYWORD = create(Color.BLUE);
    
    /** A style suitable for representing identifiers in a program. */
    public static final AttributeSet IDENTIFIER = create(Color.BLACK);
    
    /** A style suitable for representing constants in a program. */
    public static final AttributeSet LITERAL = create(MAROON);
    
    /** A style suitable for representing separating punctuation in a program. */
    public static final AttributeSet SEPARATOR = create(DARK_BLUE);
    
    /** A style suitable for representing operators in a program. */
    public static final AttributeSet OPERATOR = create(Color.BLACK, BOLD);
    
    /** A style suitable for represesenting a comment in a program. */
    public static final AttributeSet COMMENT = create(DARK_GREEN);

    /**
     * A convenience method for generating a style using regular text.
     * 
     * @param fg  the foreground color.
     * @return a style with that foreground color.
     */
    public static AttributeSet create(Color fg) {
        return create(fg, 0);
    }
    
    /**
     * A convenience method for generating a style including
     * bold/italic/both markings.
     * 
     * @param fg  the foreground color.
     * @param style  a bit mask representing bold, italic, or both,
     *    using the <code>BOLD</code> and <code>ITALIC</code> constants.
     * @return a style with the given foreground color and bold/italic markings.
     */
    public static AttributeSet create(Color fg, int style) {
        SimpleAttributeSet ret = new SimpleAttributeSet();
        StyleConstants.setForeground(ret, fg);
        StyleConstants.setBold(ret, (style & BOLD) != 0);
        StyleConstants.setItalic(ret, (style & ITALIC) != 0);
        StyleConstants.setUnderline(ret, (style & UNDERLINE) != 0);
        return ret;
    }
}
