/*
 * Copyright (c) 2005, Carl Burch.
 * 
 * This file is part of the com.cburch.editor package. The latest
 * version is available at http://www.cburch.com/proj/editor/.
 *
 * The com.cburch.editor package is free software; you can redistribute
 * it and/or modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * The com.cburch.editor package is distributed in the hope that it will
 * be useful, but WITHOUT ANY WARRANTY; without even the implied warranty
 * of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with the com.cburch.editor package; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301  USA
 */
 
 package com.cburch.editor.tokens;

import javax.swing.text.Position;

/**
 * Represents a token in the stream, holding minimal information
 * necessary for the tokenizer to work.
 * 
 * @author Carl Burch
 * @version 0.1 2005-05-31
 */
public class BasicToken {
    /** The position within the source document just past this
     * token's last character.
     */
    private Position end;
    
    /** The underlying text that this token represents. */
    private String text;

    /**
     * Constructs a token with the given information.
     * 
     * @param end  the position within the source document just past
     *     this token's last character.
     * @param text  the underlying text that this token represents.
     */
    public BasicToken(Position end, String text) {
        this.end = end;
        this.text = text;
    }
    
    /** 
     * Returns the position within the document just past this token.
     * 
     * @return the position just past this token's last character.
     */
    public final Position getEndPosition() {
        return end;
    }

    /** 
     * Returns the offset where this token's first character is
     * currently located within its document.
     * 
     * @return the offset of the first character.
     */
    public final int getBeginOffset() {
        return end.getOffset() - text.length();
    }

    /**
     * Returns the current offset of the first character following
     * this token within its document.
     * 
     * @return the offset of the first character after this token.
     */
    public final int getEndOffset() {
        return end.getOffset();
    }
    
    /**
     * Returns the number of characters in this token.
     * 
     * @return the token's length.
     */
    public final int getLength() {
        return text.length();
    }
    
    /**
     * Returns the underlying text that this token represents.
     * 
     * @return the text leading to this token.
     */
    public String getText() {
        return text;
    }
    
    /**
     * Returns a formatted string, abbreviated to the given length,
     * with any line breaks replaced with "<tt>\n</tt>." As an example of
     * the abbreviation, the text veryLargeIdentifierName would be
     * abbreviated to <tt>veryLa..me</tt> if <code>maxLength</code>
     * were 10.
     * 
     * @param maxLength  the maximum length of the returned string.
     * @return a string representing the underlying text with no
     *   line breaks, and with the middle portion removed if the
     *   text exceeds the maximum length.
     */
    public String getAbbreviatedText(int maxLength) {
        String base = text.replace("\n", "\\n");
        int len = base.length();
        if(len <= maxLength) {
            return base;
        } else if(maxLength < 6) {
            if(maxLength < 4) return base.substring(0, maxLength); 
            else              return base.substring(0, maxLength - 2) + "..";
        } else {
            return base.substring(0, maxLength - 4) + ".."
                + base.substring(len - 2);
        }
    }
    
    public String toString() {
        return "Token[" + getAbbreviatedText(20) + "]";
    }
    
    /**
     * Compares this token with the parameter. They are equal if
     * all the end offset and the text are equal.</p>
     * 
     * <p>The tokenizer uses this method to determine when it can
     * legally stop rescanning a file. If the tokens come from a
     * scanner with additional state, then the subclass should
     * override this method to make it more specific, reflecting that
     * the tokens are not equal in the eyes of the scanner.
     */
    public boolean equals(Object other) {
        if(!(other instanceof BasicToken)) return false;
        BasicToken o = (BasicToken) other;
        return end.getOffset() == o.end.getOffset()
            && this.text.equals(o.text);
    }
    
    public int hashCode() {
        return 31 * end.hashCode() + text.hashCode();
    }
}
