/*
 * Copyright (c) 2005, Carl Burch.
 * 
 * This file is part of the com.cburch.editor package. The latest
 * version is available at http://www.cburch.com/proj/editor/.
 *
 * The com.cburch.editor package is free software; you can redistribute
 * it and/or modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * The com.cburch.editor package is distributed in the hope that it will
 * be useful, but WITHOUT ANY WARRANTY; without even the implied warranty
 * of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with the com.cburch.editor package; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301  USA
 */
 
 package com.cburch.editor.util;

import java.lang.ref.WeakReference;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.NoSuchElementException;

/**
 * This class provides methods to support objects that may have
 * listeners that do not persist for the lifetime of the object.
 * It uses <code>WeakReference</code>s to remember the listeners;
 * this way, the object's existence listening to the object does
 * not count against the object's garbage collection.</p>
 * 
 * <p>You can read more about the <code>WeakReference</code> class in
 * the documentation for Java's <code>java.lang.ref</code> package.</p>
 * 
 * <p><strong>NOTE:</strong> Care must be taken with any listeners
 * that references are maintained to the listener as long as it
 * is needed. For example, the following popular idiom for creating
 * listeners is invalid.
 * <pre>
 *      // This is INVALID!!
 *      object.addListener(new Listener() {
 *           // ... required method definitions ...
 *      });
 * </pre>
 * There will be no references to the anonymous object anywhere
 * except within <code>object</code>. If <code>object</code> uses
 * <code>WeakEventSupport</code>, then this reference will not count
 * toward garbage collection. As a result, the anonymous listener
 * object is immediately eligible for garbage collection, and it
 * may well never hear of any of the events that it is intended to be
 * listening for.</p>
 * 
 * <p>In general, if you want an object to listen for events initiated
 * by an object using <code>WeakEventSupport</code>, then you will
 * want to make sure that your object maintains an instance variable
 * referring to the listener.</p>
 * 
 * @author Carl Burch
 * @version 0.1 2005-05-31
 *
 * @param <L> the type of listener we want to remember.
 */
public class WeakEventSupport<L> implements Iterable<L> {
    /**
     * Implements an iterator over the list that cleans out any
     * expired references as we go.
     */
    private class MyIterator implements Iterator<L> {
        private Iterator<WeakReference<L>> it = listeners.iterator();
        private L next = null;
        
        public boolean hasNext() {
            findNext();
            return next != null;
        }
        
        public L next() {
            findNext();
            if(next == null) throw new NoSuchElementException();
            L ret = next;
            next = null;
            return ret;
        }
        
        public void remove() {
            throw new UnsupportedOperationException();
        }
        
        private void findNext() {
            if(next != null) return;
            while(it.hasNext()) {
                next = it.next().get();
                if(next != null) return;
                it.remove();
            }
        }
    }
    
    private LinkedList<WeakReference<L>> listeners = new LinkedList<WeakReference<L>>();

    /**
     * Constructs an empty list of listeners.
     */
    public WeakEventSupport() { }

    /**
     * Adds a weak reference to the given listener into our list.
     * 
     * @param listener  the listener to be added.
     */
    public void add(L listener) {
        listeners.add(new WeakReference<L>(listener));
    }
    
    /**
     * Removes the given listener from our list.
     * 
     * @param listener  the listener to be removed.
     */
    public void remove(L listener) {
        for(Iterator<WeakReference<L>> it = listeners.iterator(); it.hasNext(); ) {
            L l = it.next().get();
            if(l == null || l == listener) it.remove();
        }
    }
    
    /**
     * Says whether the list currently contains any elements that
     * have not been collected out. It is possible that that
     * <code>isEmpty</code> will return <code>false</code> at one
     * time and <code>true</code> later on, even if no listeners have
     * been explicitly added, because some listeners may have
     * been stale and thus collected out.
     * 
     * @return <code>true</code> if there are currently no active
     *     listeners in the list; <code>false</code> otherwise.
     */
    public boolean isEmpty() {
        for(Iterator<WeakReference<L>> it = listeners.iterator(); it.hasNext(); ) {
            L l = it.next().get();
            if(l != null) return false;
            it.remove();
        }
        return true;
    }
    
    /**
     * Returns the number of active listeners in the list. This
     * number may decrease, even when there are no deletions,
     * because some listeners may become stale and be collected out.
     *   
     * @return the number of listeners in the list.
     */
    public int size() {
        for(Iterator<WeakReference<L>> it = listeners.iterator(); it.hasNext(); ) {
            L l = it.next().get();
            if(l == null) it.remove();
        }
        return listeners.size();
    }
    
    /**
     * Returns an iterator for stepping through the active listeners
     * of the list.
     * 
     * @return the iterator over active listeners.
     */
    public Iterator<L> iterator() {
        return new MyIterator();
    }
}
