/* Copyright (c) 2003, Carl Burch. License information is located in the
 * edu.csbsju.socs.grammar.Main source code and at
 * www.cburch.com/proj/grammar/. */

package edu.csbsju.socs.grammar;

import java.awt.*;
import java.awt.event.*;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;
import java.util.StringTokenizer;
import javax.swing.*;
import javax.swing.event.*;
import edu.csbsju.socs.util.*;

class EditorFrame extends JFrame implements LocaleManager.Listener {
	private static final String HELP_DIR = "edu/csbsju/socs/grammar/doc/";
	private static Font USER_FONT = new Font("monospaced", Font.PLAIN, 12);
	class FileMenu extends JMenu implements ActionListener {
		JMenuItem newi  = new JMenuItem();
		JMenuItem open  = new JMenuItem();
		JMenuItem save  = new JMenuItem();
		JMenuItem print = new JMenuItem();
		JMenuItem quit  = new JMenuItem();

		FileMenu() {
			add(newi);  newi.addActionListener(this);
			add(open);  open.addActionListener(this);
			add(save);  save.addActionListener(this);
			addSeparator();
			add(print); print.addActionListener(this);
			add(quit);  quit.addActionListener(this);
		}

		public void renewStrings() {
			this.setText(Strings.get("fileMenu"));
			newi.setText(Strings.get("newMenuItem"));
			open.setText(Strings.get("openMenuItem"));
			save.setText(Strings.get("saveMenuItem"));
			print.setText(Strings.get("printMenuItem"));
			quit.setText(Strings.get("quitMenuItem"));
		}

		public void actionPerformed(ActionEvent evt) {
			Object src = evt.getSource();
			if(src == newi) doNew();
			else if(src == open) doOpen();
			else if(src == save) doSave();
			else if(src == print) doPrint();
			else if(src == quit) doQuit();
		}

		private void doNew() {
			if(!confirmLoss(Strings.get("confirmLossMsg"))) return;
			editor.setText("");
			setChanged(false);
		}

		private void doOpen() {
			if(!confirmLoss(Strings.get("confirmLossMsg"))) return;

			int val = chooser.showOpenDialog(this);
			if(val != JFileChooser.APPROVE_OPTION) return;

			StringBuffer buffer = new StringBuffer();
			try {
				java.io.BufferedReader reader = new java.io.BufferedReader(
					new java.io.FileReader(chooser.getSelectedFile()));
				while(true) {
					String line = reader.readLine();
					if(line == null) break;
					buffer.append(line);
					buffer.append("\n");
				}
				reader.close();
			} catch(java.io.IOException e) {
				JOptionPane.showMessageDialog(this,
					Strings.get("saveError") + e.getMessage());
				return;
			}

			editor.setText(buffer.toString());
			setChanged(false);
		}

		private void doPrint() {
			try {
				PrintUtilities.printText(editor.getText());
			} catch(java.awt.print.PrinterException e) {
				JOptionPane.showMessageDialog(EditorFrame.this,
					Strings.get("printError") + e);
			}
		}
	}

	class EditMenu extends JMenu implements ActionListener {
		JMenuItem cut = new JMenuItem();
		JMenuItem copy = new JMenuItem();
		JMenuItem paste = new JMenuItem();

		EditMenu() {
			add(cut);   cut.addActionListener(this);
			add(copy);  copy.addActionListener(this);
			add(paste); paste.addActionListener(this);
		}

		public void renewStrings() {
			this.setText(Strings.get("editMenu"));
			cut.setText(Strings.get("cutMenuItem"));
			copy.setText(Strings.get("copyMenuItem"));
			paste.setText(Strings.get("pasteMenuItem"));
		}

		public void actionPerformed(ActionEvent evt) {
			Object src = evt.getSource();
			if(src == cut)        editor.cut();
			else if(src == copy)  editor.copy();
			else if(src == paste) editor.paste();
		}
	}

	class FontSizeItem extends JRadioButtonMenuItem
			implements ActionListener {
		private int size;
		FontSizeItem(int size, ButtonGroup bgroup) {
			this.size = size;
			bgroup.add(this);
			addActionListener(this);
			setSelected(size == USER_FONT.getSize());
		}
		public void actionPerformed(ActionEvent evt) {
			if(isSelected()) {
				Font old = editor.getFont();
				if(size != old.getSize()) {
					Font newf = old.deriveFont((float) size);
					editor.setFont(newf);
					controls.field.setFont(newf);
					EditorFrame.this.pack();
				}
			}
		}
	}

	class OptionsMenu extends JMenu {
		JMenu size = new JMenu();
		FontSizeItem[] size_items;
		OptionsMenu() {
			ButtonGroup bgroup = new ButtonGroup();
			int[] sizes = { 12, 14, 16, 18, 20, 24 };
			size_items = new FontSizeItem[sizes.length];
			for(int i = 0; i < sizes.length; i++) {
				size_items[i] = new FontSizeItem(sizes[i], bgroup);
				size.add(size_items[i]);
			}
			add(size);
			JMenuItem item = Strings.createLocaleMenuItem();
			if(item != null) add(item);
		}
		public void renewStrings() {
			this.setText(Strings.get("optionsMenu"));
			size.setText(Strings.get("fontSizeMenuItem"));
			for(int i = 0; i < size_items.length; i++) {
				size_items[i].setText(size_items[i].size
					+ " " + Strings.get("fontSizeUnits"));
			}
		}
	}

	class HelpMenu extends JMenu
			implements ActionListener {
		private JMenuItem help = new JMenuItem();
		private JMenuItem about = new JMenuItem();
		java.net.URL index = null;

		HelpMenu() {
			add(help);  help.addActionListener(this);
			add(about); about.addActionListener(this);
		}
		public void renewStrings() {
			this.setText(Strings.get("helpMenu"));
			help.setText(Strings.get("helpMenuItem"));
			about.setText(Strings.get("aboutMenuItem"));

			help_frame.setTitle(Strings.get("helpTitle"));
			help_frame.clearContentsItems();
			try {
				index = help_frame.getURL(HELP_DIR
					+ Strings.get("helpOverviewLoc"));
				help_frame.setLocation(index);
				help_frame.addContentsItem(Strings.get("helpOverview"),
					index);
				help_frame.addContentsItem(Strings.get("helpSyntax"),
					help_frame.getURL(HELP_DIR + Strings.get("helpSyntaxLoc")));
			} catch(HelpFrame.NotFoundException e) { }
		}
		public void actionPerformed(ActionEvent event) {
			if(event.getSource() == help) {
				try {
					if(index == null) {
						index = help_frame.getURL(HELP_DIR
							+ Strings.get("helpOverviewLoc"));
					}
					help_frame.load(index);
					help_frame.show();
				} catch(HelpFrame.NotFoundException e) {
					JOptionPane.showMessageDialog(EditorFrame.this,
						Strings.get("helpNotFoundError") + ": "
						+ e.getMessage());
				}
			} else if(event.getSource() == about) {
				JOptionPane.showMessageDialog(EditorFrame.this,
					Strings.get("aboutMessage"));
			}
		}
	}

	private class ControlPanel extends JPanel implements ActionListener {
		private JButton parse = new JButton();
		private JButton generate = new JButton();
		private JButton clear = new JButton();
		private JLabel field_label = new JLabel();
		private JTextField field = new JTextField();

		ControlPanel() {
			super(new BorderLayout());

			JPanel controls = new JPanel();
			controls.add(parse);    parse.addActionListener(this);
			controls.add(generate); generate.addActionListener(this);
			controls.add(clear);    clear.addActionListener(this);

			field_label.setBorder(BorderFactory.createMatteBorder(5, 5,
				0, 0, getBackground()));
			add(field_label, BorderLayout.WEST);
			field.addActionListener(this);
			field.setFont(USER_FONT);
			field.setBorder(BorderFactory.createCompoundBorder(
				BorderFactory.createMatteBorder(5, 5, 0, 5, getBackground()),
				field.getBorder()));
			add(field, BorderLayout.CENTER);
			add(controls, BorderLayout.SOUTH);
		}

		public void renewStrings() {
			parse.setText(Strings.get("parseButton"));
			generate.setText(Strings.get("generateButton"));
			clear.setText(Strings.get("clearButton"));
			parse.setToolTipText(Strings.get("parseTip"));
			generate.setToolTipText(Strings.get("generateTip"));
			clear.setToolTipText(Strings.get("clearTip"));
			field_label.setText(Strings.get("textLabel"));
		}

		public void actionPerformed(ActionEvent e) {
			Object src = e.getSource();
			if(src == parse || src == field) doParse();
			if(src == generate) doGenerate();
			if(src == clear) field.setText("");
		}

		private void doGenerate() {
			Grammar g = getGrammar(); 
			if(g == null) return;
			List sentence;
			try {
				sentence = Generator.generate(g);
			} catch(Generator.GenerateException e) {
				JOptionPane.showMessageDialog(EditorFrame.this,
					Strings.get("generateError") + e.getMessage());
				return;
			}
			StringBuffer buf = new StringBuffer();
			Iterator it = sentence.iterator();
			while(it.hasNext()) {
				if(buf.length() > 0) buf.append(" ");
				buf.append(it.next().toString());
			}
			field.setText(buf.toString());
		}

		private void doParse() {
			// first get Chomsky normal form
			Grammar g = getGrammar();
			if(g == null) return;
			StringTokenizer toks = new StringTokenizer(field.getText());
			Grammar.Atom[] text = new Grammar.Atom[toks.countTokens()];
			boolean text_ok = true;
			for(int i = 0; toks.hasMoreTokens(); i++) {
				text[i] = g.getAtom(toks.nextToken());
				if(text[i] == null) text_ok = false;
			}
			Tree t = null;
			if(text_ok) {
				if(parser == null) parser = new Parser(g);
				t = parser.parse(text);
			}
			tree_frame.setTree(t);
			field.selectAll();
			field.requestFocus();
			tree_frame.show();
		}

		private Grammar getGrammar() {
			try {
				if(grammar == null) {
					grammar = GrammarParser.parse(editor.getText());
				}
				return grammar;
			} catch(GrammarParser.ParseException e) {
				JOptionPane.showMessageDialog(EditorFrame.this,
					e.toString());
				return null;
			}
		}
	}

	class MyDocumentListener implements DocumentListener {
        public void insertUpdate(DocumentEvent evt) {
			setChanged(true);
        }
        public void removeUpdate(DocumentEvent evt) {
			setChanged(true);
        }
        public void changedUpdate(DocumentEvent evt) {
			setChanged(true);
        }
    }

	private class WindowCloser extends WindowAdapter {
		public void windowClosing(WindowEvent e) {
			doQuit();
		}
	}

	private FileMenu file_menu = new FileMenu();
	private EditMenu edit_menu = new EditMenu();
	private OptionsMenu options_menu = new OptionsMenu();
	private HelpMenu help_menu = new HelpMenu();

	private JLabel editor_label = new JLabel("", SwingConstants.LEFT);
	private JTextArea editor = new JTextArea(4, 20);
	private ControlPanel controls = new ControlPanel();

	private JFileChooser chooser = new JFileChooser(".");
	private TreeFrame tree_frame = new TreeFrame();
	private HelpFrame help_frame = new HelpFrame();
	private boolean changed = false;
	private Grammar grammar = null;
	private Parser parser = null;

	EditorFrame() {
		addWindowListener(new WindowCloser());

		JMenuBar menubar = new JMenuBar();
		menubar.add(file_menu);
		menubar.add(edit_menu);
		menubar.add(options_menu);
		menubar.add(help_menu);
		setJMenuBar(menubar);

		editor.setFont(USER_FONT);
		editor.getDocument().addDocumentListener(new MyDocumentListener());
		JScrollPane scroll = new JScrollPane(editor);
		scroll.setBorder(BorderFactory.createCompoundBorder(
			BorderFactory.createMatteBorder(0, 5, 5, 5, getBackground()),
			scroll.getBorder()));
		editor_label.setBorder(BorderFactory.createMatteBorder(5, 5,
			0, 5, getBackground()));

		Container contents = getContentPane();
		contents.add(editor_label, BorderLayout.NORTH);
		contents.add(scroll, BorderLayout.CENTER);
		contents.add(controls, BorderLayout.SOUTH);
		LocaleManager.addListener(this);
		localeChanged();
	}

	public void localeChanged() {
		file_menu.renewStrings();
		edit_menu.renewStrings();
		options_menu.renewStrings();
		help_menu.renewStrings();
		controls.renewStrings();
		tree_frame.renewStrings();
		help_frame.renewStrings();
		this.setTitle(Strings.get("editorTitle"));
		editor_label.setText(Strings.get("grammarLabel"));
		pack();

		JFileChooser newch = new JFileChooser();
		java.io.File f = chooser.getCurrentDirectory();
		if(f != null) newch.setCurrentDirectory(f);
		f = chooser.getSelectedFile();
		if(f != null) newch.setSelectedFile(f);
		chooser = newch;
	}

	private void setChanged(boolean value) {
		changed = value;
		grammar = null;
		parser = null;
	}

	private boolean doSave() {
		int val = chooser.showSaveDialog(this);
		if(val != JFileChooser.APPROVE_OPTION) return false;

		try {
			java.io.PrintWriter writer = new java.io.PrintWriter(
				new java.io.FileWriter(chooser.getSelectedFile()));
			writer.print(editor.getText());
			writer.close();
		} catch(java.io.IOException e) {
			JOptionPane.showMessageDialog(this,
				Strings.get("saveError") + e.getMessage());
			return false;
		}
		setChanged(false);
		return true;
	}

	private void doQuit() {
		if(!confirmLoss(Strings.get("confirmQuitMsg"))) return;
		System.exit(0);
	}

	private boolean confirmLoss(String msg) {
		if(!changed) return true;
		toFront();
		int val = JOptionPane.showConfirmDialog(this, msg);
		if(val == JOptionPane.CANCEL_OPTION) return false;
		if(val == JOptionPane.NO_OPTION && !doSave()) return false;
		return true;
	}
}
