/* Copyright (c) 2003, Carl Burch. License information is located in the
 * edu.csbsju.socs.grammar.Main source code and at
 * www.cburch.com/proj/grammar/. */

package edu.csbsju.socs.grammar;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Set;

class Generator {
	private static final int MAX_DEPTH = 50;
	private static final int MAX_TRIES = 50;
	private static java.util.Random rand = new java.util.Random();

	public static class GenerateException extends Exception {
		public GenerateException(String msg) { super(msg); }
	}

	public static List generate(Grammar g) throws GenerateException {
		int try_count = 0;
		boolean verified = false;
		Grammar.Symbol root = g.getRoot();
		while(try_count < MAX_TRIES) {
			++try_count;
			try {
				ArrayList log = new ArrayList();
				generateSub(g.getRoot(), 0, g, log);
				return log;
			} catch(GenerateException e) {
				// ok, we must have gone too deep. See whether
				// generation is even possible
				if(!verified) {
					if(!terminates(g)) {
						throw new GenerateException(Strings.get("generateEmptyError"));
					}
					verified = true;
				}
			}
		}
		throw new GenerateException(Strings.get("generateFailedError"));
	}
	private static void generateSub(Grammar.Symbol root, int depth,
			Grammar g, ArrayList log) throws GenerateException {
		if(depth > MAX_DEPTH) {
			throw new GenerateException("maximum depth exceeded");
		}
		Collection rules = g.getRules(root);

		int which = rand.nextInt(rules.size());
		Iterator it = rules.iterator();
		while(it.hasNext()) {
			Grammar.Rule rule = (Grammar.Rule) it.next();
			if(which == 0) {
				Grammar.Element[] rhs = rule.getRightSide();
				for(int i = 0; i < rhs.length; i++) {
					Grammar.Element e = rhs[i];
					if(e instanceof Grammar.Symbol) {
						generateSub((Grammar.Symbol) e, depth + 1,
							g, log);
					} else {
						log.add(e);
					}
				}
				break;
			}
			--which;
		}
	}

	private static boolean terminates(Grammar g) {
		// determine which symbols terminate in some string
		// (cf. page 89, Hopcroft and Ullman)
		HashSet terminating = new HashSet();
		boolean changed = true;
		while(changed) {
			changed = false;
			Iterator it = g.getRules().iterator();
			while(it.hasNext()) {
				Grammar.Rule rule = (Grammar.Rule) it.next();
				Grammar.Symbol lhs = rule.getLeftSide();
				if(terminating.contains(lhs)) continue;
				if(allSymsInSet(rule.getRightSide(), terminating)) {
					terminating.add(lhs);
					changed = true;
				}
			}
		}
		return terminating.contains(g.getRoot());
	}
	private static boolean allSymsInSet(Grammar.Element[] rhs, Set q) {
		for(int i = 0; i < rhs.length; i++) {
			if(rhs[i] instanceof Grammar.Symbol
					&& !q.contains(rhs[i])) {
				return false;
			}
		}
		return true;
	}
}
