/*
 * Word.java
 *
 * Created on February 15, 2004, 1:16 PM
 */
package DataRep;
import javax.swing.*;
/**
 *
 * @author  lziegler
 */
public abstract class Word extends JPanel {
    
    /**
     * class Bit represents a single binary digit - the
     * fundamental component of the word. It is an inner
     * class since it is a button that, when clicked, can
     * change the value of the Word.
     */
    class Bit extends JButton implements java.awt.event.ActionListener {
        
        /**
         * This creates an instance of Bit as part of a word. The parameter
         * position tells where the Bit will appear in the word - bit 0 being
         * the leftmost, bit 1 the next, ...
         */
        public Bit(int position) {
            super();
            if((position<wordLength()) && (position>=0)) {
                this.setText("0");
                this.setMargin(new java.awt.Insets(2,2,2,2));
                this.position=position;
                this.setFont(new java.awt.Font("Dialog",1, fontSize));
                this.setBackground(new java.awt.Color(0,0,255));
                this.setForeground(new java.awt.Color(255,255,0));
                orMask=orMask<<(wordLength()-position-1);
                andMask=andMask-orMask;
                addActionListener(this);
            }
            else {
                System.out.println("position "+position+
                "is illegal for a word of size "+
                wordLength());
                System.exit(0);
            }
            
        }
        
        /**
         * actionPerformed is what happens when the Bit button is clicked.
         * Basically, the value of the word is changed by setting this bit
         * in the long integer containing the value for the word to the
         * opposite value from what it was and then changing the displayed
         * value for the word.
         */
        public void actionPerformed(java.awt.event.ActionEvent e) {
            if(getText().equals("0")) {
                setText("1");
                wordValue=wordValue | orMask;
                this.setBackground(java.awt.Color.yellow);
                this.setForeground(new java.awt.Color(0,0,255));
            }
            else {
                setText("0");
                this.setBackground(new java.awt.Color(0,0,255));
                this.setForeground(java.awt.Color.yellow);
                wordValue=wordValue & andMask;
            }
            display();
        }
        
        /**
         * setBit(value) sets this bit to the bit at this position in the
         * long word value.
         */
        public void setBit(long value) {
            if((value & orMask) == 0) {
                setText("0");
                this.setBackground(new java.awt.Color(0,0,255));
                this.setForeground(java.awt.Color.yellow);
            }
            else {
                setText("1");
                this.setBackground(java.awt.Color.yellow);
                this.setForeground(new java.awt.Color(0,0,255));
            }
        }
        
        /**
         * position is the position in the word for this bit.
         */
        private int position;
        
        /**
         * orMask is used to set the corresponding bit in the word's value
         * to 1.
         */
        public long orMask=1;
        
        /**
         * andMask is used to set the corresponding bit in the word's value
         * to 0.
         */
        public long andMask=-1;
    }
    
    /**
     * These constants - SIGNED, UNSIGNED, OCTAL, HEX, REAL, and CHAR - are
     * included to make it simple to determine the value of the data type
     * being displayed. They are static so that they can be used outside the
     * class as well.
     */
    public static final int SIGNED=0;
    public static final int UNSIGNED=1;
    public static final int OCTAL=2;
    public static final int HEX=3;
    public static final int REAL=4;
    public static final int CHAR=5;
    
    /**
     * The default constructor simply calls the constructor for a JPanel.
     */
    public Word() {
        super();
    }
    
    /**
     * This constructor creates a new instance of Word. The first parameter,
     * value, is the value that the word and its bits are set to. The second,
     * fontSize, determines the font size and the overall size of the displayed
     * word and its bits. spaceMode determines whether we are in student
     * learning mode (bit layout unspecified) or in teaching mode (bit layout
     * correlates with the datatype). Finally, the dataType is one of SIGNED,
     * UNSIGNED, OCTAL, HEX, REAL, or CHAR. (CHAR only works for 8 bit words
     * and UNSIGNED does not work for 64 bit words.)
     *
     * The constructor also lays out the bits and text field properly and sets
     * up other necessary constants. Note we use a null layout for exact
     * positioning.
     */
    public Word(long value, int fontSize, boolean spaceMode, int dataType) {
        super();
        this.wordValue=value;
        this.fontSize=fontSize;
        this.dataType=dataType;
        this.spaceMode=spaceMode;
        setSpacing();
        bits=new Bit[wordLength()];
        int spaceIndex=0;
        bitPanels[spaceIndex]=new javax.swing.JPanel(new java.awt.FlowLayout(java.awt.FlowLayout.CENTER,0,5));
        for(int i=0;i<wordLength();i++) {
            bits[i]=new Bit(i);
            bits[i].setFont(new java.awt.Font("Dialog", 0, fontSize));
            bitPanels[spaceIndex].add(bits[i]);
            if(i==bitSpacing[spaceIndex]) {
                add(bitPanels[spaceIndex]);
                javax.swing.JLabel temp=new javax.swing.JLabel(" ");
                temp.setSize(fontSize,fontSize);
                add(temp);
                spaceIndex++;
                bitPanels[spaceIndex]=new JPanel(new java.awt.FlowLayout(java.awt.FlowLayout.CENTER,0,5));
            }
        }
        valueField=new javax.swing.JTextField();
        valueField.setFont(new java.awt.Font("Dialog",0,fontSize));
        int columnFactor=10;
        switch(dataType) {
            case CHAR: columnFactor=5; break;
            case REAL:
                if(wordLength()==64) columnFactor=22;
                else columnFactor=12;
                break;
            case UNSIGNED: case SIGNED: columnFactor=(wordLength()+6)/3; break;
            case OCTAL: columnFactor=(wordLength()+6)/3; break;
            case HEX: columnFactor=(wordLength()+8)/4; break;
        }
        valueField.setColumns(columnFactor);
        valueField.setHorizontalAlignment(javax.swing.JTextField.RIGHT);
        valueField.setText("0");
        valueField.addFocusListener(new java.awt.event.FocusAdapter() {
            public void focusLost(java.awt.event.FocusEvent evt) {
                valueFieldFocusLost(evt);
            }
        });
        valueField.addActionListener(new java.awt.event.ActionListener() {
            public void actionPerformed(java.awt.event.ActionEvent evt) {
                valueFieldActionPerformed(evt);
            }
        });
        valueField.addMouseListener(new java.awt.event.MouseAdapter() {
            public void mouseExited(java.awt.event.MouseEvent evt) {
                valueFieldMouseExited(evt);
            }
        });
        valueField.setFont(new java.awt.Font("Dialog", 0, fontSize));
        bitPanels[spaceIndex].add(valueField);
        bitPanels[spaceIndex].setVisible(true);
        add(bitPanels[spaceIndex]);
        setBits(value);
        display();
    }
    
    /**
     * When the mosuse leaves the text field after a value is entered in
     * the text field, this method will insure that the display and values
     * get properly updated.
     */
    private void valueFieldMouseExited(java.awt.event.MouseEvent evt) {
        setBits(valueField.getText());
        display();
    }
    
    /**
     * When a return is hit after a value is entered in the text field,
     * this method will insure that the display and values get properly
     * updated.
     */
    private void valueFieldActionPerformed(java.awt.event.ActionEvent evt) {
        setBits(valueField.getText());
        display();
    }
    
    /**
     * When the user moves out of focus on the text field after changing
     * vthe alue, this method will insure that the display and values get
     * properly updated.
     */
    private void valueFieldFocusLost(java.awt.event.FocusEvent evt) {
        setBits(valueField.getText());
        display();
    }
    
    /**
     * getDouble() returns the double value represented by the bits of the
     * Word.
     */
    public double getDouble() {
        return(computeDouble());
    }
    
    /**
     * getLong() returns the long integer value represented by the bits of
     * the Word.
     */
    public long getLong() {
        return(wordValue);
    }
    
    /**
     */
    public int getDataType() {
        return dataType;
    }
    
    /**
     * retrieveHeight() returns the height required to properly display
     * the Word.
     */
    public abstract int retrieveHeight();
    
    
    /**
     * retrieveWidth() returns the width required to properly display
     * the Word.
     */
    public abstract int retrieveWidth();
    
    /**
     * wordLength() returns the length of this Word.
     */
    public abstract int wordLength();
    
    /**
     * setBits(value) sets the Word value and its bits to the specified
     * long value and adjusts the display.
     */
    public void setBits(long value){
        for(int i=0;i<wordLength();i++) {
            bits[i].setBit(value);
        }
        wordValue=value;
        display();
    }
    /**
     * setBits(value) sets the Word value and its bits to the specified
     * double value and adjusts the display.
     */
    public abstract void setDoubleBits(double value);
    
    public abstract int getLargeFont();
    
    public abstract int getMediumFont();
    
    public abstract int getSmallFont();
    
    /**
     * setBits(value) sets the Word value and its bits to the specified
     * String valueFieldValue according to the type being represented and
     * then adjusts the display.
     */
    public abstract void setBits(String valueFieldValue);
    
    
    /**
     * computeDouble returns the double value represented by the bits of
     * the Word.
     */
    public abstract double computeDouble();
    
    /**
     * display() changes the text field to properly display the information
     * stored in the word.
     */
    public abstract void display();
    
    /**
     * setSpacing() creates the array bitSpacing which is used to insert
     * spaces in the Bit display in the right spots. The spacing depends
     * on the values of wordSize, dataType, and spaceMode.
     */
    public abstract void setSpacing();
    
    /**
     * valueField is the text field used to display the value in the
     * correct dataType.
     */
    protected JTextField valueField;
    
    /**
     * This is the array of bit buttons used in the display.
     */
    protected Bit [] bits;
    
    protected JPanel [] bitPanels=new JPanel[34];
    
    /**
     * bitSpzcing contains the array of positions for adding space in the
     * display of the bits in the Word.
     */
    protected int [] bitSpacing;
    
    /**
     * wordValue is the long value containing the actual value of the words.
     */
    public long wordValue;
    
    /**
     * dataType contains the data type to be displayed (CHAR, SIGNED, UNSIGNED,
     * HEX, OCTAL, or REAL).
     */
    protected int dataType;
    
    /**
     * fontSize is the font Size for the Word display.
     */
    protected int fontSize;
    
    /**
     * spaceMode is true if used for teaching mode, false if used for student
     * mode. Default is student mode.
     */
    protected boolean spaceMode=false;
    
    /**
     * overallWidth contains the overall width needed to display the Word.
     */
    protected int overallWidth;
    
    /**
     * overallHeight contains the overall height needed to display the Word.
     */
    protected int overallHeight;
    
}
